<?php

namespace Drupal\next_tag_revalidator\Plugin\Next\Revalidator;

use Drupal\Core\Form\FormStateInterface;
use Drupal\next\Event\EntityActionEvent;
use Drupal\next\Plugin\ConfigurableRevalidatorBase;
use Drupal\next\Plugin\RevalidatorInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides cache tag-based revalidation for Next.js ISR pages.
 *
 * Revalidates Next.js pages based on cache tags when content changes.
 *
 * @Revalidator(
 *  id = "nextjs_cache_tag",
 *  label = "Next.js Cache Tag",
 *  description = "Cache tag-based revalidation for Next.js ISR pages"
 * )
 */
class CacheTagRevalidator extends ConfigurableRevalidatorBase implements RevalidatorInterface {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'entity_tag' => TRUE,
      'entity_list_tag' => TRUE,
      'additional_tags' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get entity type and bundle from form callback object.
    $entity_type_id = NULL;
    $bundle = NULL;

    try {
      $entity_bundle_string = $form_state->getBuildInfo()['callback_object']->getEntity()->id();
      // Split "node.quote" into entity type and bundle.
      if (strpos($entity_bundle_string, '.') !== FALSE) {
        [$entity_type_id, $bundle] = explode('.', $entity_bundle_string, 2);
      }
    }
    catch (\Exception) {
      // Fallback if we can't get the entity info.
      $entity_type_id = NULL;
      $bundle = NULL;
    }

    $form['entity_tag'] = [
      '#title' => $this->t('Revalidate entity cache tag'),
      '#description' => $this->t('Revalidate pages with the individual entity cache tag (e.g., @entity_type:123).', [
        '@entity_type' => $entity_type_id ?: 'node',
      ]),
      '#type' => 'checkbox',
      '#default_value' => $this->configuration['entity_tag'] ?? TRUE,
    ];

    // Generate specific label and description based on detected entity type.
    if ($entity_type_id && $bundle) {
      if ($entity_type_id === 'node') {
        $list_tag_example = 'node_list:' . $bundle;
        $next_js_example = 'tags: ["node_list:' . $bundle . '"]';
      }
      elseif ($entity_type_id === 'taxonomy_term') {
        $list_tag_example = 'taxonomy_list:' . $bundle;
        $next_js_example = 'tags: ["taxonomy_list:' . $bundle . '"]';
      }
      else {
        $list_tag_example = $entity_type_id . '_list:' . $bundle;
        $next_js_example = 'tags: ["' . $entity_type_id . '_list:' . $bundle . '"]';
      }

      $title = $this->t('Revalidate @tag cache tags', ['@tag' => $list_tag_example]);
      $description = $this->t('Revalidates pages tagged with @tag when @entity_type entities of type @bundle change.<br><br>In Next.js use: <code>@example</code>', [
        '@tag' => $list_tag_example,
        '@entity_type' => $entity_type_id,
        '@bundle' => $bundle,
        '@example' => $next_js_example,
      ]);
    }
    else {
      $title = $this->t('Revalidate [entity_type]_list:[bundle] cache tags');
      $description = $this->t('Revalidates pages tagged with entity type and bundle list cache tags when entities change.<br><strong>Node entities:</strong> generates node_list:[bundle] (e.g., node_list:article, node_list:person)<br><strong>Taxonomy terms:</strong> generates taxonomy_list:[vocabulary] (e.g., taxonomy_list:tags)<br><strong>Other entities:</strong> generates [entity_type]_list:[bundle]<br><br>In Next.js use: <code>tags: ["node_list:article"]</code> or <code>tags: ["taxonomy_list:tags"]</code>');
    }

    $form['entity_list_tag'] = [
      '#title' => $title,
      '#description' => $description,
      '#type' => 'checkbox',
      '#default_value' => $this->configuration['entity_list_tag'] ?? TRUE,
    ];

    $form['additional_tags'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional cache tags to revalidate'),
      '#default_value' => $this->configuration['additional_tags'] ?? '',
      '#description' => $this->t('Additional cache tags to revalidate when this entity type changes. Enter one tag per line. Examples:<br>node_list:all<br>search_results<br>homepage'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['entity_tag'] = $form_state->getValue('entity_tag');
    $this->configuration['entity_list_tag'] = $form_state->getValue('entity_list_tag');
    $this->configuration['additional_tags'] = $form_state->getValue('additional_tags');
  }

  /**
   * {@inheritdoc}
   */
  public function revalidate(EntityActionEvent $event): bool {
    $revalidated = FALSE;

    $sites = $event->getSites();
    if (!count($sites)) {
      return FALSE;
    }

    $entity = $event->getEntity();
    $cache_tags = [];

    // Add individual entity cache tag.
    if (!empty($this->configuration['entity_tag'])) {
      $cache_tags[] = $entity->getEntityTypeId() . ':' . $entity->id();
    }

    // Add entity list cache tag based on entity type.
    if (!empty($this->configuration['entity_list_tag'])) {
      $entity_type_id = $entity->getEntityTypeId();
      $bundle = $entity->bundle();

      // Generate appropriate list tag based on entity type.
      if ($entity_type_id === 'node') {
        $cache_tags[] = 'node_list:' . $bundle;
      }
      elseif ($entity_type_id === 'taxonomy_term') {
        $cache_tags[] = 'taxonomy_list:' . $bundle;
      }
      // Add support for other entity types if needed.
      else {
        $cache_tags[] = $entity_type_id . '_list:' . $bundle;
      }
    }

    // Add additional cache tags.
    if (!empty($this->configuration['additional_tags'])) {
      $additional_tags = array_map('trim', explode("\n", $this->configuration['additional_tags']));
      $additional_tags = array_filter($additional_tags);
      $cache_tags = array_merge($cache_tags, $additional_tags);
    }

    if (!count($cache_tags)) {
      return FALSE;
    }

    // Remove duplicates.
    $cache_tags = array_unique($cache_tags);

    foreach ($cache_tags as $cache_tag) {
      /** @var \Drupal\next\Entity\NextSite $site */
      foreach ($sites as $site) {
        try {
          $revalidate_url = $site->buildRevalidateUrl(['tags' => $cache_tag]);

          if (!$revalidate_url) {
            throw new \Exception('No revalidate url set.');
          }

          if ($this->nextSettingsManager->isDebug()) {
            $this->logger->notice('(@action): Revalidating cache tag %tag for entity %entity_type:%entity_id on site %site. URL: %url', [
              '@action' => $event->getAction(),
              '%tag' => $cache_tag,
              '%entity_type' => $entity->getEntityTypeId(),
              '%entity_id' => $entity->id(),
              '%site' => $site->label(),
              '%url' => $revalidate_url->toString(),
            ]);
          }

          $response = $this->httpClient->request('GET', $revalidate_url->toString());
          if ($response->getStatusCode() === Response::HTTP_OK) {
            if ($this->nextSettingsManager->isDebug()) {
              $this->logger->notice('(@action): Successfully revalidated cache tag %tag for entity %entity_type:%entity_id on site %site.', [
                '@action' => $event->getAction(),
                '%tag' => $cache_tag,
                '%entity_type' => $entity->getEntityTypeId(),
                '%entity_id' => $entity->id(),
                '%site' => $site->label(),
              ]);
            }

            $revalidated = TRUE;
          }
        }
        catch (\Exception $exception) {
          $this->logger->error('Cache tag revalidation failed: @message', [
            '@message' => $exception->getMessage(),
          ]);
        }
      }
    }

    return $revalidated;
  }

}
