# Next.js Tag Revalidator

A Drupal module that provides **configurable** cache tag-based revalidation for Next.js ISR (Incremental Static Regeneration) pages.

## Why This Module?

The core [Next.js module](https://www.drupal.org/project/next) includes a basic "Cache Tag" revalidator, but it has some limitations for advanced use cases:

- **No configuration options** - Always revalidates ALL entity cache tags and list cache tags
- **No selectivity** - Cannot choose which types of cache tags to revalidate
- **Performance impact** - Sends many unnecessary cache tags in a single request
- **Limited control** - No way to add custom cache tags or modify behavior
- **Generic list tags** - Uses `getListCacheTags()` which generates generic tags like `node_list` rather than bundle-specific tags like `node_list:article` that Next.js applications typically need for efficient cache invalidation

This module provides:

- ✅ **Granular control** - Choose individual entity tags, list tags, or both
- ✅ **Custom cache tags** - Add your own cache tags for specific use cases  
- ✅ **Context-aware UI** - Configuration form shows relevant examples for your content types
- ✅ **Performance optimized** - Only revalidate the cache tags you actually need
- ✅ **Individual tag requests** - Sends one HTTP request per cache tag (better for debugging)
- ✅ **Flexible configuration** - Different settings per entity type and bundle

## Features

- **Automatic cache tag generation** - Intelligently generates cache tags based on entity type and bundle
- **Individual entity tags** - Revalidates pages tagged with specific entity IDs (e.g., `node:123`)
- **Entity list tags** - Revalidates pages tagged with entity type/bundle combinations:
  - Nodes: `node_list:article`, `node_list:page`
  - Taxonomy terms: `taxonomy_list:tags`, `taxonomy_list:categories`
  - Other entities: `{entity_type}_list:{bundle}`
- **Custom additional tags** - Configure extra cache tags to revalidate
- **Visual configuration UI** - Context-aware form with Next.js code examples

## Requirements

- Drupal 10 or 11
- [Next.js module](https://www.drupal.org/project/next) for Drupal

## Installation

1. Place this module in `modules/custom/` or `modules/contrib/`
2. Enable the module: `drush en next_tag_revalidator`
3. Configure revalidation rules in your Next.js site settings

## Configuration

1. Go to **Configuration > Web Services > Next.js sites**
2. Edit your Next.js site
3. Add a new revalidator and select "Next.js Cache Tag"
4. Configure which cache tags to revalidate:
   - **Entity cache tag**: Individual entity revalidation (e.g., `node:123`)
   - **Entity list cache tag**: Bundle-specific list revalidation (e.g., `node_list:article`)
   - **Additional cache tags**: Custom tags (one per line)

## Usage in Next.js

Tag your Next.js pages/components with the appropriate cache tags:

```javascript
// For individual entities
export default async function Article({ params }) {
  const article = await fetch(`${drupalUrl}/jsonapi/node/article/${params.id}`, {
    next: { tags: [`node:${params.id}`] }
  });
  // ...
}

// For entity lists
export default async function ArticleList() {
  const articles = await fetch(`${drupalUrl}/jsonapi/node/article`, {
    next: { tags: ['node_list:article'] }
  });
  // ...
}

// For custom tags
export default async function HomePage() {
  const data = await fetch(`${drupalUrl}/api/homepage-data`, {
    next: { tags: ['homepage', 'featured_content'] }
  });
  // ...
}
```

## How It Works

When content changes in Drupal:

1. The module detects entity create/update/delete events
2. Generates appropriate cache tags based on configuration
3. Calls your Next.js revalidation endpoint with the cache tags
4. Next.js invalidates and regenerates pages tagged with those cache tags

## Cache Tag Patterns

| Entity Type | Individual Tag | List Tag |
|-------------|---------------|----------|
| Node | `node:123` | `node_list:article` |
| Taxonomy Term | `taxonomy_term:456` | `taxonomy_list:tags` |
| User | `user:789` | `user_list:user` |
| Custom Entity | `my_entity:101` | `my_entity_list:my_bundle` |

## Debugging

Enable debug mode in your Next.js site configuration to see revalidation requests in the Drupal logs.

## Contributing

This module is designed to be a generic solution for Next.js cache tag revalidation. Please submit issues and pull requests to help improve it for the community.

## License

GPL-2.0+