<?php

declare(strict_types=1);

namespace Drupal\nextgen\Drush\Generators;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\ContentEntityType;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use DrupalCodeGenerator\Asset\AssetCollection as Assets;
use DrupalCodeGenerator\Attribute\Generator;
use DrupalCodeGenerator\GeneratorType;
use Symfony\Component\DependencyInjection\ContainerInterface;

#[Generator(
  name: 'next-entity-component',
  description: 'Generate Next.js entity component.',
  templatePath: __DIR__,
  type: GeneratorType::OTHER,
)]

/**
 * Next generator for view pages.
 */
final class EntityComponentGenerator extends NextgenBase implements ContainerInjectionInterface {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityTypeBundleInfoInterface $entityTypeBundleManager,
    private readonly EntityDisplayRepositoryInterface $entityDisplayManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly RendererInterface $renderer,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_display.repository'),
      $container->get('entity_field.manager'),
      $container->get('renderer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function generate(array &$vars, Assets $assets): void {
    $ir = $this->createInterviewer($vars);

    // Get content entity type.
    $entities = $this->getContentEntities();
    if (empty($entities)) {
      $this->io()->error((string) $this->t('Content entities list is empty'));
      return;
    }
    $entity_type = $ir->choice('Entity type', $entities);

    // Get entity bundles.
    $entity_bundles = $this->getEntityBundles($entity_type);
    if ($entity_bundles) {
      $entity_bundle = $ir->choice('Entity bundle', $entity_bundles,
        default: count($entity_bundles) == 1 ? current($entity_bundles) : NULL
      );
    }
    else {
      $entity_bundle = $entity_type;
    }

    // Get display nodes.
    $entity_display_modes = $this->getDisplayModes($entity_type, $entity_bundle);
    $entity_display_mode = $ir->choice('Entity view mode', $entity_display_modes,
      default: count($entity_display_modes) == 1 ? current($entity_display_modes) : NULL
    );

    // Render fields.
    $vars['fields'] = $this->renderFields($entity_type, $entity_bundle, $entity_display_mode);

    // Get class value.
    $vars['type'] = $entity_type;
    $vars['bundle'] = $entity_bundle;
    $vars['display'] = $entity_display_mode === 'default' ? '' : $entity_display_mode;
    $vars['class'] = $ir->askClass(default: '{type|camelize}{bundle|camelize}{display|camelize}');

    // Get component path.
    $vars['path'] = $ir->ask('Component path', default: $entity_type);

    // Create directory and file.
    $dir_path = $this->nextdir($vars, 'components', $assets);
    if ($dir_path) {
      $assets->addFile($dir_path . '/' . $vars['class'] . '.tsx', 'component.twig');
    }
  }

  /**
   * Get content entities.
   *
   * @return array
   *   Content entities list.
   */
  private function getContentEntities(): array {
    $content_entity_types = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type instanceof ContentEntityType) {
        $content_entity_types[$entity_type_id] = (string) $entity_type->getLabel();
      }
    }

    return $content_entity_types;
  }

  /**
   * Get entity bundles.
   *
   * @param string $entity_type_id
   *   Entity type ID.
   *
   * @return ?array
   *   Entity bundles list.
   */
  private function getEntityBundles(string $entity_type_id): ?array {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id, FALSE);

    if ($entity_type && $entity_type->hasKey('bundle')) {
      $bundles = $this->entityTypeBundleManager->getBundleInfo($entity_type_id);
      return array_map(fn($bundle) => $bundle['label'], $bundles);
    }

    return NULL;
  }

  /**
   * Get entity bundle's display modes.
   *
   * @param string $entity_type_id
   *   Entity type ID.
   * @param string $entity_bundle
   *   Entity bundle ID.
   *
   * @return array
   *   Entity bundle display mode list.
   */
  private function getDisplayModes(string $entity_type_id, string $entity_bundle): array {
    $view_modes = $this->entityDisplayManager->getViewModes($entity_type_id);

    $bundle_display['default'] = (string) $this->t('Default');
    foreach ($view_modes as $machine_name => $info) {
      $display = $this->entityDisplayManager->getViewDisplay($entity_type_id, $entity_bundle, $machine_name);
      if ($display && !$display->isNew()) {
        $bundle_display[$machine_name] = $info['label'];
      }
    }

    return $bundle_display;
  }

  /**
   * Get fields list for entity bundle's display modes.
   *
   * @param string $entity_type_id
   *   Entity type ID.
   * @param string $entity_bundle
   *   Entity bundle ID.
   * @param string $view_mode
   *   View mode.
   *
   * @return array
   *   Fields list.
   */
  private function getViewModeFields(string $entity_type_id, string $entity_bundle, string $view_mode): array {
    $display = $this->entityDisplayManager->getViewDisplay($entity_type_id, $entity_bundle, $view_mode);
    if (!$display || $display->isNew()) {
      return [];
    }

    return $display->getComponents();
  }

  /**
   * Return fields.
   *
   * @param string $entity_type_id
   *   Entity type ID.
   * @param string $entity_bundle
   *   Entity bundle ID.
   * @param string $entity_display_mode
   *   View mode.
   *
   * @return array
   *   Rendered fields.
   */
  private function renderFields(string $entity_type_id, string $entity_bundle, string $entity_display_mode) {
    $allowed_types = nextgen_get_allowed_types();
    $fields = $this->getViewModeFields($entity_type_id, $entity_bundle, $entity_display_mode);
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $entity_bundle);

    $render_fields = [];
    foreach ($fields as $field_name => $field) {
      $type = $field['type'] ?? NULL;
      if ($type && in_array($type, $allowed_types)) {
        $build = [
          '#theme' => 'nextgen_' . $type,
          '#field' => $field_name,
          '#label_position' => $field['label'],
          '#label' => (string) $field_definitions[$field_name]->getLabel(),
        ];

        $render_fields[] = [
          'name' => $field_name,
          'render' => $this->renderer->renderInIsolation($build),
        ];
      }
    }

    return $render_fields;
  }

}
