<?php

declare(strict_types=1);

namespace Drupal\nextgen\Drush\Generators;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use DrupalCodeGenerator\Asset\AssetCollection as Assets;
use DrupalCodeGenerator\Attribute\Generator;
use DrupalCodeGenerator\GeneratorType;
use Symfony\Component\DependencyInjection\ContainerInterface;

#[Generator(
  name: 'next-view-page',
  description: 'Generate Next.js pages from Views pages.',
  templatePath: __DIR__,
  type: GeneratorType::OTHER,
)]

/**
 * Next generator for view pages.
 */
final class PageViewGenerator extends NextgenBase implements ContainerInjectionInterface {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self($container->get('entity_type.manager'));
  }

  /**
   * {@inheritdoc}
   */
  protected function generate(array &$vars, Assets $assets): void {
    $ir = $this->createInterviewer($vars);

    // Get views ID.
    $view_storage = $this->entityTypeManager->getStorage('view');
    $view_ids = $view_storage
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('status', 1)
      ->execute();

    if (empty($view_ids)) {
      $this->io()->warning((string) $this->t('The site has no views.'));
      return;
    }

    $vars['view'] = $ir->choice('View ID', $view_ids,
      default: count($view_ids) == 1 ? current($view_ids) : NULL
    );
    $view = $view_storage->load($vars['view']);

    // Get page displays of the view.
    $page_displays = [];
    foreach ($view->get('display') as $display_id => $display) {
      if (!empty($display['display_plugin']) && $display['display_plugin'] === 'page') {
        $page_displays[$display_id] = $display_id;
      }
    }

    if (empty($page_displays)) {
      $this->io()->warning((string) $this->t('The view "@view" has no page displays.', [
        '@view' => $vars['view'],
      ]));
      return;
    }

    $vars['display'] = $ir->choice('View page display', $page_displays,
      default: count($page_displays) == 1 ? current($page_displays) : NULL
    );

    // Get display parameters.
    $display = $view->getDisplay($vars['display']);
    $display_options = $display['display_options'];
    $default_display_options = $view->getDisplay('default')['display_options'];
    $display_options = array_replace_recursive($default_display_options, $display_options);

    // Get page's path.
    $path = $display_options['path'] ?? NULL;
    if (!$path) {
      $this->io()->error((string) $this->t("The view display has not the 'Path' setting."));
      return;
    }
    $vars['path'] = $ir->ask('Page path', default: $path);

    // Get page's title.
    $title = $display_options['title'];
    $vars['title'] = $ir->ask('Page title', default: $title);

    // Get pager options.
    $display_pager = $display_options['pager'];
    $vars['pager_type'] = $display_pager['type'];
    $vars['pager_items_per_page'] = $display_pager['options']['items_per_page'] ?? 10;

    // Get exposed filters, based on taxonomy.
    $exposed_filters = $this->getExposedFilters($display_options);
    if ($exposed_filters) {
      $vars['exposed'] = $exposed_filters;
    }

    // Get class value.
    $vars['class'] = $ir->askClass(default: '{view|camelize}{display|camelize}');

    // Create directory and file.
    $dir_path = $this->nextdir($vars, 'app', $assets);
    if ($dir_path) {
      $assets->addFile("$dir_path/page.tsx", 'view-page.twig');
    }
  }

  /**
   * Get Exposed filter data.
   *
   * @param array $display_options
   *   Display options.
   *
   * @return array|bool
   *   Exposed filter data.
   */
  protected function getExposedFilters($display_options) {
    $filters = $display_options['filters'] ?? NULL;
    if ($filters) {
      $data = [];
      foreach ($filters as $filter) {
        $expose = $filter['expose'] ?? NULL;
        if ($expose && $filter['plugin_id'] === 'taxonomy_index_tid') {
          $data[] = [
            'id' => $expose['identifier'],
            'label' => $expose['label'],
            'vid' => $filter['vid'],
          ];
        }
      }
      return $data;
    }
  }

}
