<?php
/**
 * @file
 * Contains \Drupal\noahs_popups\Controller\NoahsPopupsBuildController.
 */

namespace Drupal\noahs_popups\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\node\Entity\NodeType;
use Drupal\editor\Entity\Editor;
use \Drupal\Component\Render\FormattableMarkup;
use Drupal\noahs_page_builder\ModalForm;
use Drupal\noahs_page_builder\Controller\NoahsController;
use Drupal\noahs_page_builder\Controller\NoahsSaveStylesController;
use Drupal\noahs_page_builder\Service\WidgetServices;
use Drupal\noahs_page_builder\Service\ControlServices;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Database\Connection;


class NoahsPopupsBuildController extends ControllerBase{

   
  /**
   * The widget services.
   *
   * @var \Drupal\noahs_page_builder\Service\WidgetServices
   */
  protected $widgetService;
   
  /**
   * The widget controls.
   *
   * @var \Drupal\noahs_page_builder\Service\ControlServices
   */
  protected $controlService;

  /**
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The widget services.
   *
   * @var \Drupal\noahs_page_builder\ModalForm
   */

   protected $modalForm;

   /**
   * The database connection.
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;    

  /**
   * Constructor.
   *
   * @param \Drupal\noahs_page_builder\Service\WidgetServices $widget_service
   *   The widget service.
   * @param \Drupal\noahs_page_builder\Service\ControlServices $control_service
   *   The control service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\noahs_page_builder\ModalForm $modal_form
   *   The control manager service.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   */
  public function __construct(
    WidgetServices $widget_service,
    ControlServices $control_service,
    EntityTypeManagerInterface $entity_type_manager,
    ModalForm $modal_form,
    Connection $connection = NULL
  ) {
    $this->widgetService = $widget_service;
    $this->controlService = $control_service;
    $this->entityTypeManager = $entity_type_manager;
    $this->modalForm = $modal_form;
    $this->connection = $connection ?: \Drupal::database();
  }

  /**
   * Creates an instance of the controller.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   Returns an instance of this controller.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('noahs_page_builder.widget_service'),
      $container->get('noahs_page_builder.control_service'),
      $container->get('entity_type.manager'),
      $container->get('noahs_page_builder.modal_form'),
      $container->get('database')
    );
  }

  /**   * Builds the Noahs theme editor page.
   * @param string $type
   *   The type of the Noahs theme.
   * @return array
   *   Returns the render array for the Noahs theme editor page.
   */
  public function build($noahs_editor, $type) {

    $widgets = $this->widgetService->getWidgetsList();
    $data = $this->loadNoahsTheme($type) ?? NULL;
    $getPageUrl =  Url::fromRoute('noahs_popups.save_theme', [],  ['absolute' => TRUE])->toString();
    $all_widgets = $this->widgetService->getAllWidgetsList();
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    $popup_settings = null;
    $page_settings = null;

    if (!empty($data->page_settings)) {
      $page_settings = !empty($data->page_settings) ? json_decode($data->page_settings, true) : array('settings' => []);
    }

    if (!empty($data->popup_settings)) {
      $popup_settings = $data->popup_settings;
    }
    
    $fields = [];
    foreach ($all_widgets as $k => $el) {
      $fields[$k] = $this->widgetService->getWidgetFields($k);
    }
    $default_fields = $this->controlService->defaultFields();
    $page_fields = $this->pageSettingsForm();


    $merge_page_fields = array_merge($page_fields, $default_fields);

    $token = \Drupal::service('csrf_token')->get('');
    $page['#attached']['drupalSettings']['csrfToken'] = $token;
    $page['#attached']['drupalSettings']['noahs_items_fields'] = $fields;
    $page['#attached']['drupalSettings']['noahs_page_fields'] = $merge_page_fields;
    $page['#attached']['drupalSettings']['savePage'] = $getPageUrl;
    $page['#attached']['drupalSettings']['all_widgets'] = $all_widgets;
    $page['#attached']['drupalSettings']['theme_builder'] = true;
    $page['#attached']['drupalSettings']['langcode'] = $langcode;
    $page['#attached']['drupalSettings']['theme_builder_type'] = $type;
    $page['#attached']['drupalSettings']['entity_type'] = 'popup';
    $page['#attached']['drupalSettings']['noahs_id'] = $type;
    $page['#attached']['library'][] = 'noahs_page_builder/noahs_page_builder.assets.preview';
    $page['#attached']['library'][] = 'noahs_popups/noahs_popups.assets.editor';
    $page['#attached']['drupalSettings']['popup_settings'] = $popup_settings;
 
    // Ckeditor drupal settings
    $plugin_manager = \Drupal::service('plugin.manager.editor');

    // Ckeditor drupal settings
    $noahs_page_builder_config = \Drupal::config('noahs_page_builder.settings');
    if(!empty($noahs_page_builder_config->get('use_editor'))){

      $editor = Editor::load($noahs_page_builder_config->get('use_editor'));
      $plugin = $plugin_manager->createInstance($editor->getEditor());
      $plugin_definition = $plugin->getPluginDefinition();
      $settings = $plugin->getJSSettings($editor);

      $page['#attached']['library'] = array_merge($page['#attached']['library'], $plugin->getLibraries($editor));
      $page['#attached']['drupalSettings']['noahs_page_builder']['ckeditor_settings'] = $settings;
    }

    $nodos = $this->entityTypeManager->getStorage('node')->loadMultiple();
    $pages = [];

    foreach ($nodos as $node) {
      $pages[] = [
        'id' => $node->id(),
        'title' => $node->label(),
      ];
    }

    $tipos = NodeType::loadMultiple();
    $content_types = [];

    foreach ($tipos as $content_type) {
      $content_types[] = [
        'id' => $content_type->id(),
        'label' => $content_type->label(),
      ];
    }

    $page['#attached']['drupalSettings']['noahs_poups']['pages'] = $pages;
    $page['#attached']['drupalSettings']['noahs_poups']['content_types'] = $content_types;
    $url = Url::fromRoute('module_new', [], ['absolute' => TRUE]);

    $iframe_url = Url::fromRoute('noahs_popups.iframe', 
    [
      'noahs_preview' => 'preview-popup',
      'type' => $type
    ],
    ['absolute' => TRUE])->toString();

    $page['noahs-popup-theme-form'] = array(
      '#theme' => 'noahs-popup-theme-form',
      '#type' => $type,
      '#entity_type' => 'popup',
      '#widgets' => $widgets,
      '#iframe_url' => $iframe_url,
      '#page_settings_form' => $this->renderSettingsForm($type),
      '#page_settings' => !empty($page_settings['page_settings']) ? $page_settings['page_settings'] :  null,
      '#popup_settings' => isset($popup_settings) ? json_decode($popup_settings, true) :  null,
    );

    return $page;

  }

  /**   * Builds the iframe for the Noahs theme.
   * @param string $type
   *   The type of the Noahs theme.
   * @return array
   *   Returns the render array for the iframe.
   */

  public function iframe($noahs_preview, $type){

    $entity_type = 'popup';
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    $widgets = $this->widgetService->getWidgetsList();
    $sections = null;
    $data = $this->loadNoahsTheme($type) ?? NULL;
    $page_settings = !empty($data->page_settings) ? json_decode($data->page_settings, true) : [];

    if(!empty($data->settings)){
      $sections = noahs_page_builder_get_sections($data->settings);
    }

    $html = noahs_page_builder_html_generated($sections);

    $globlal_widgets = noahs_page_builder_pro_load_widgets();
    $defaults_widgets = noahs_page_builder_pro_load_defaults_widgets();  
    $getPageUrl =  Url::fromRoute('noahs_page_builder_pro.save_theme', [],  ['absolute' => TRUE])->toString();
    $module_url =  '/'.\Drupal::service('extension.list.module')->getPath('noahs_page_builder');
    
    $page['#attached']['drupalSettings']['savePage'] = $getPageUrl;
    $page['#attached']['drupalSettings']['page_settings'] = $page_settings;
    $page['#attached']['drupalSettings']['entity_type'] = 'pro_theme';
    // $page['#attached']['drupalSettings']['noahs_page_builder']['classes'] = $classes;
    $page['#attached']['drupalSettings']['uid'] = \Drupal::currentUser()->id();
    $page['#attached']['drupalSettings']['langcode'] = $langcode;
    $page['#attached']['drupalSettings']['noahs_id'] = $type;
    $page['#attached']['drupalSettings']['entity_type'] = $entity_type;
    $page['#attached']['library'][] = 'noahs_page_builder/noahs_page_builder.assets.admin';
    $page['#attached']['library'][] = 'noahs_popups/noahs_popups.assets.preview';


    $page['noahs-popup-preview'] = [
      '#content' => '',
      '#theme' => 'noahs-popup-preview',
      '#page_settings' => $page_settings,
      '#entity_type' => $entity_type,
      '#type' => $type,
      '#langcode' => $langcode,
      '#widgets' => $widgets,
      '#noahs_pro' => is_module_installed('noahs_page_builder_pro'),
      '#generated_html' => $html,
    ];
    return $page;
  
  }


  /**   * Save the popup settings.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object containing the popup settings.
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Returns a JSON response indicating the result of the save operation.
   */

  public function savePopupSettings(Request $request){
    $data = json_decode($request->getContent(), TRUE);
    $noahs_id = $data['noahs_id'];
    $settings = $data['settings'];
    $page_settings = $data['page_settings'] ?? [];
    $popup_settings = $data['popup_settings'] ?? [];
    $langcode = $data['langcode'] ?? \Drupal::languageManager()->getCurrentLanguage()->getId();
    $css = $data['css'] ?? '';

    $builder = \Drupal::database()->select('noahs_page_builder_popups', 'd')
      ->fields('d', array('noahs_id'))
      ->condition('noahs_id', $noahs_id)
      ->condition('langcode', $langcode)
      ->execute()
      ->fetchAssoc();
    
    if($builder != NULL){
        $builder = \Drupal::database()->update("noahs_page_builder_popups")
        ->fields(array(
            'settings' => $settings,
            'page_settings' => $page_settings,
            'popup_settings' => $popup_settings,
            'modified_date' => time(),
        ))
        ->condition('noahs_id', $noahs_id)
        ->execute();
        $result = 'Plantilla Actualizada correctamente';
    }else{
        $builder = \Drupal::database()->insert("noahs_page_builder_popups")
          ->fields(array(
              'noahs_id' => $noahs_id,
              'langcode' => $langcode,
              'settings' => $settings,
              'page_settings' => $page_settings,
              'popup_settings' => $popup_settings,
              'modified_date' => time(),
          ))
        ->execute();
        $result = 'Plantilla añadida correctamente';
    }
    $noahsContrller = \Drupal::classResolver(NoahsController::class);
    $css = $noahsContrller->saveCss('popups', $noahs_id, $langcode, $css);

    return new JsonResponse($result); 
  }

  /**
   * Load the Noahs theme by ID.
   *
   * @param string $noahs_id
   *   The ID of the Noahs theme.
   *
   * @return object|null
   *   Returns the Noahs theme object or NULL if not found.
   */
  public function loadNoahsTheme($noahs_id) {
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    $result = \Drupal::database()->select('{noahs_page_builder_popups}', 'd')
    ->fields('d')
    ->condition('noahs_id', $noahs_id, '=')
    ->execute()
    ->fetchObject();

    return $result;
  }

  public function renderPopup($noahs_id) {

    $sections = null;
    $data = $this->loadNoahsTheme($noahs_id) ?? NULL;

    if(!empty($data->settings)){
      $sections = noahs_page_builder_get_sections($data->settings);
    }

    $html = '<div class="noahs-popup-element" id="noahs-popup-' .$data->noahs_id.'">';
    $html .= '<div class="builder-wrapper noahs-theme-content-wrapper noahs-popups-builder-wrapper scrollme">';
    $html .= '<button type="button" class="noahs_close_popup"><i class="fa-solid fa-xmark"></i></button>';
    $html .= noahs_page_builder_html_generated($sections);
    $html .= '</div>';

    return new Response($html);
  }

  private function pageSettingsForm() {

    $form = [];
    // Section Content.
    $form['popup_section_content'] = [
      'type' => 'tab',
      'title' => t('Content'),
    ];

    $form['popup_width'] = [
      'type'    => 'noahs_width',
      'title'   => t('Width'),
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'max-width',
      'tab' => 'popup_section_content',
      'responsive' => TRUE,
      'placeholder' => 'use as 10%, 100px, 100vw...',
      'wrapper' => FALSE,
      'open' => TRUE,
      'slide' => TRUE,
      'units' => []
    ];

    $form['popup_height'] = [
      'type'    => 'text',
      'title'   => t('Height'),
      'tab' => 'popup_section_content',
      'placeholder'     => t('Height'),
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'min-height',
    ];
    
    $form['popup_max_height'] = [
      'type'    => 'text',
      'title'   => t('Max Height'),
      'tab' => 'popup_section_content',
      'placeholder'     => t('Max Height'),
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'max-height',
    ];

    $form['popup_position'] = [
      'type'    => 'select',
      'title'   => t('Position'),
      'tab' => 'popup_section_content',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'position',
      'responsive' => TRUE,
      'options' => [
        'absolute' => t('Default'),
        'fixed' => 'Fixed',
      ],
    ];
    $form['popup_position_coordinates'] = [
      'type'    => 'noahs_coordinates',
      'fields'    => [
        'left' => [
          'title' => t('Left'),
          'property' => 'left',
        ],
        'top' => [
          'title' => t('Top'),
          'property' => 'top',
        ],
        'right' => [
          'title' => t('Right'),
          'property' => 'right',
        ],
        'bottom' => [
          'title' => t('Bottom'),
          'property' => 'bottom',
        ],
      ],
      'title'   => ('Position Coordinates'),
      'tab' => 'popup_section_content',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'position',
      'responsive' => TRUE,
    ];


    // Section Styles.
    // Styles.
    $form['popup_section_styles'] = [
      'type' => 'tab',
      'title' => t('Styles'),
    ];

    $form['popup_group'] = [
      'type' => 'group',
      'tab' => 'popup_section_styles',
      'title' => t('Popup Styles'),
    ];

    $form['popup_wrapper_padding'] = [
      'type' => 'noahs_padding',
      'title' => t('Wrapper Padding'),
      'tab' => 'popup_section_styles',
      'group' => 'popup_group',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'padding',
      'responsive' => TRUE,
      'style_hover' => TRUE,
    ];

    $form['bg_color'] = [
      'type'     => 'noahs_color',
      'title'    => ('Background Color'),
      'tab'     => 'popup_section_styles',
      'style_type' => 'style',
      'style_css' => 'background-color',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'group' => 'popup_group',
    ];

    $form['popup_border'] = [
      'type' => 'noahs_border',
      'title' => t('Border'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'border',
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'popup_group',
    ];

    $form['popup_padding'] = [
      'type' => 'noahs_padding',
      'title' => t('Padding'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'padding',
      'default_value' => [
        'padding_left' => '25px',
        'padding_right' => '25px',
        'padding_top' => '25px',
        'padding_bottom' => '25px',
      ],
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'popup_group',
    ];

    $form['popup_shadows'] = [
      'type'    => 'noahs_shadows',
      'title'   => t('Image Shadow'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'popup_group',
    ];

    $form['popup_border'] = [
      'type' => 'noahs_border',
      'title' => t('Border'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_css' => 'border',
      'style_hover' => TRUE,
      'group' => 'popup_group',
    ];

    $form['popup_radius'] = [
      'type'    => 'noahs_radius',
      'title'   => t('Border Radius'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'popup_group',
    ];

    $form['popup_bg_image'] = [
      'type'     => 'noahs_background_image',
      'title'    => ('Background Image'),
      'tab'     => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'style_hover' => TRUE,
      'responsive' => TRUE,
      'group' => 'popup_group',
    ];

    $form['popup_bg_gradient'] = [
      'type'     => 'noahs_background_gradient',
      'title'    => ('Background Gradient'),
      'tab'     => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => '.noahs-popups-builder-wrapper',
      'responsive' => TRUE,
      'group' => 'popup_group',
    ];

    // Btn Close
    $form['btn_close_group'] = [
      'type' => 'group',
      'tab' => 'popup_section_styles',
      'title' => t('Button close'),
    ];

    $form['btn_close_font'] = [
      'type' => 'noahs_font',
      'title' => t('Font'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => 'button.noahs_close_popup',
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'btn_close_group',
    ];

    $form['btn_close_bg_color'] = [
      'type'     => 'noahs_color',
      'title'    => ('Background Color'),
      'tab'     => 'popup_section_styles',
      'style_type' => 'style',
      'style_css' => 'background-color',
      'style_selector' => 'button.noahs_close_popup',
      'group' => 'btn_close_group',
    ];

    $form['btn_close_popup_border'] = [
      'type' => 'noahs_border',
      'title' => t('Border'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => 'button.noahs_close_popup',
      'style_css' => 'border',
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'btn_close_group',
    ];

    $form['btn_close_popup_padding'] = [
      'type' => 'noahs_padding',
      'title' => t('Padding'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => 'button.noahs_close_popup',
      'style_css' => 'padding',
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'btn_close_group',
    ];

    $form['btn_close_popup_shadows'] = [
      'type'    => 'noahs_shadows',
      'title'   => t('Image Shadow'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => 'button.noahs_close_popup',
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'btn_close_group',
    ];

    $form['btn_close_popup_border'] = [
      'type' => 'noahs_border',
      'title' => t('Border'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => 'button.noahs_close_popup',
      'style_css' => 'border',
      'style_hover' => TRUE,
      'group' => 'btn_close_group',
    ];

    $form['btn_close_popup_radius'] = [
      'type'    => 'noahs_radius',
      'title'   => t('Border Radius'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => 'button.noahs_close_popup',
      'responsive' => TRUE,
      'style_hover' => TRUE,
      'group' => 'btn_close_group',
    ];

    $form['btn_close_coordinates'] = [
      'type'    => 'noahs_coordinates',
      'fields'    => [
        'left' => [
          'title' => t('Left'),
          'property' => 'left',
        ],
        'top' => [
          'title' => t('Top'),
          'property' => 'top',
        ],
        'right' => [
          'title' => t('Right'),
          'property' => 'right',
        ],
        'bottom' => [
          'title' => t('Bottom'),
          'property' => 'bottom',
        ],
      ],
      'title'   => ('Position Coordinates'),
      'tab' => 'popup_section_styles',
      'style_type' => 'style',
      'style_selector' => 'button.noahs_close_popup',
      'style_css' => 'position',
      'responsive' => TRUE,
      'group' => 'btn_close_group',
    ];

    // Overlay Styles.
    $form['overlay_group'] = [
      'type' => 'group',
      'tab' => 'popup_section_styles',
      'title' => t('Overlay Styles'),
    ];

    $form['overlay_bg_color'] = [
      'type'     => 'noahs_color',
      'title'    => ('Background Color'),
      'tab'     => 'popup_section_styles',
      'style_type' => 'style',
      'style_css' => 'background-color',
      'style_selector' => '.noahs-popup-element',
      'group' => 'overlay_group',
    ];
    return $form;
  }

  private function renderSettingsForm($type) {
    // Render the page settings form.
    $data = $this->loadNoahsTheme($type) ?? [];

    $page_settings = !empty($data->page_settings) ? json_decode($data->page_settings, true) : [];
    // Load the data for the page settings form.

    $form = $this->pageSettingsForm();

    $renderForm = $this->modalForm->renderForm($form, $page_settings, null, '', ['section_extras']);
    return $renderForm;
  }

}

