<?php

namespace Drupal\noahs_popups\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\Core\Database\Database;
use Symfony\Component\HttpFoundation\RedirectResponse;

class NoahsPopupListController extends ControllerBase {

  public function list() {
    $header = [
      'name' => $this->t('Name'),
      'Language' => $this->t('Language'),
      'status' => $this->t('Status'),
      'operations' => $this->t('Operations'),
    ];

    $rows = [];
    $connection = Database::getConnection();
    $results = $connection->select('noahs_page_builder_popups', 'p')
      ->fields('p', ['name', 'noahs_id', 'langcode', 'status'])
      ->execute();

    foreach ($results as $record) {
      $native_language = \Drupal::languageManager()->getNativeLanguages()[$record->langcode] ?? FALSE;
      $edit_url = Url::fromRoute('noahs_popups.build', [
        'noahs_editor' => 'popup-builder',
        'type' => $record->noahs_id,
      ]);

      $toggle_url = Url::fromRoute('noahs_popups.toggle', ['noahs_id' => $record->noahs_id]);
      $delete_url = Url::fromRoute('noahs_popups.delete', ['noahs_id' => $record->noahs_id]);

      $is_active = (int) $record->status === 1;
      $status_badge = [
        '#type' => 'inline_template',
        '#template' => '<span class="badge {{ cls }}" data-popup-status="{{ status }}">{{ label }}</span>',
        '#context' => [
          'cls' => $is_active ? 'badge--success' : 'badge--warning',
          'status' => $is_active ? '1' : '0',
          'label' => $is_active ? $this->t('Active') : $this->t('Inactive'),
        ],
      ];

      $links = [
        'edit' => [
          'title' => $this->t('Edit'),
          'url' => $edit_url,
        ],
        'toggle' => [
          'title' => $is_active ? $this->t('Deactivate') : $this->t('Activate'),
          'url' => $toggle_url,
          'attributes' => [
            'class' => ['button', 'button--small'],
            'data-action' => 'toggle',
            'data-noahs-id' => $record->noahs_id,
          ],
        ],
        'delete' => [
          'title' => $this->t('Delete'),
          'url' => $delete_url,
          'attributes' => [
            'class' => ['use-ajax', 'button', 'button--danger'],
            'data-dialog-type' => 'modal',
            'data-dialog-options' => '{"width":480}',
          ],
        ],
      ];

      $rows[] = [
        'data' => [
          ['data' => $record->name],
          ['data' => ($native_language) ? $native_language->getName() : 'Unknown'],
          ['data' => $status_badge],
          ['data' => [
            '#type' => 'operations',
            '#links' => $links,
            '#dropbutton' => TRUE,
          ]],
        ],
      ];
    }

    $build = [];

    $add_url = Url::fromRoute('noahs_popups.create');
    $build['add'] = [
      '#type' => 'link',
      '#title' => $this->t('➕ Añadir nuevo popup'),
      '#url' => $add_url,
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'style' => 'margin-bottom: 20px; display: inline-block;',
      ],
    ];

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No hay popups creados aún.'),
    ];

    return $build;
  }

  public function toggle($noahs_id) {
    $connection = Database::getConnection();
    $record = $connection->select('noahs_page_builder_popups', 'p')
      ->fields('p', ['status'])
      ->condition('noahs_id', $noahs_id)
      ->execute()
      ->fetchAssoc();

    if ($record) {
      $new = ((int) $record['status'] === 1) ? 0 : 1;
      $connection->update('noahs_page_builder_popups')
        ->fields(['status' => $new])
        ->condition('noahs_id', $noahs_id)
        ->execute();
      $this->messenger()->addStatus($new ? $this->t('Popup activated.') : $this->t('Popup deactivated.'));
    }
    else {
      $this->messenger()->addError($this->t('Popup not found.'));
    }

    return new RedirectResponse(Url::fromRoute('noahs_popups.popup_list')->toString());
  }
}
