<?php

namespace Drupal\node_authlink\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessCheck;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\Routing\Route;

/**
 * Class NodeRevisionAccessCheck.
 *
 * @package Drupal\node_authlink\Access
 */
class NodeRevisionAccessCheck extends EntityAccessCheck {

  /**
   * Perform actual check on a node given the account and the operation.
   *
   * @param \Drupal\node\NodeInterface $node
   * @param \Drupal\Core\Session\AccountInterface $account
   * @param mixed $op
   *
   * @return bool
   *   True if the module Node Authlink has found an authorization to perform the
   *   given operation on the given node for the given account.
   */
  public function checkAccess(NodeInterface $node, AccountInterface $account, $op = 'view') {
    return !$node->isDefaultRevision() && node_authlink_node_is_enabled($node) && node_authlink_check_authlink($node, $op, $account);
  }

  /**
   * {@inheritdoc}
   */
  public function access(Route $route, RouteMatchInterface $route_match, AccountInterface $account) {

    // Split the entity type and the operation.
    /** @var string $requirement */
    $requirement = $route->getRequirement('_entity_access');
    [$entity_type, $operation] = explode('.', $requirement);

    $parameters = $route_match->getParameters();
    $node = $parameters->has($entity_type) ? $parameters->get($entity_type) : NULL;

    if ($node instanceof NodeInterface
      && $entity_type === 'node_revision'
      && $this->checkAccess($node, $account, $operation)) {
      return AccessResult::allowed();
    }

    return parent::access($route, $route_match, $account);
  }

}
