<?php

namespace Drupal\node_authlink\Form;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;

/**
 * Provides a confirmation form for deleting a node authlink.
 */
class NodeAuthlinkDeleteForm extends ConfirmFormBase {

  /**
   * The node containing the authlink to be deleted.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_authlink_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete the authorization link for %title?', [
      '%title' => $this->node->label()
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return Url::fromRoute('view.node_authlinks.admin_page');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {
    $this->node = $node;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    try {
      node_authlink_delete($this->node);
      $this->messenger()->addStatus($this->t('The authorization link for %title has been deleted.', [
        '%title' => $this->node->label()
      ]));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('An error occurred while deleting the authorization link.'));
    }

    $form_state->setRedirect('view.node_authlinks.admin_page');
  }

  /**
   * Access callback for the delete form.
   */
  public function access(AccountInterface $account, NodeInterface $node = NULL) {
    if ($node instanceof NodeInterface) {
      return AccessResult::allowedIf(
        $account->hasPermission('create and delete node authlinks') ||
        $account->hasPermission(sprintf('create and delete node %s authlinks', $node->bundle()))
      );
    }
    return AccessResult::forbidden();
  }
}
