<?php

namespace Drupal\node_authlink\Form;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatter;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Utility\LinkGenerator;
use Drupal\node\Entity\Node;
use Drupal\node\NodeInterface;
use Drupal\node_authlink\ConfigurationHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Class NodeAuthlinkNodeForm.
 *
 * This is the form of the authlinks for a given node.
 */
class NodeAuthlinkNodeForm extends FormBase {


  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config_factory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Link generator service.
   *
   * @var \Drupal\Core\Utility\LinkGenerator
   */
  protected $linkGenerator;

  /**
   * Date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatter
   */
  protected $dateFormatter;

  /**
   * Constructs a new NodeAuthlinkNodeForm object.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *  Config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *  Entity type manager.
   * @param \Drupal\Core\Utility\LinkGenerator $link_generator
   *   Link Generator.
   * @param \Drupal\Core\Datetime\DateFormatter $date_formatter
   *   Date Formatter.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    LinkGenerator $link_generator,
    DateFormatter $date_formatter
  ) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->linkGenerator = $link_generator;
    $this->dateFormatter = $date_formatter;
    }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('link_generator'),
      $container->get('date.formatter'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_authlink_node_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {

    // This should never happen but let's play safe.
    if (!($node instanceof NodeInterface)) {
      $form['no_grants'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('Error: empty node received.') . '</p>',
      ];
      return $form;
    }

    $bundle = $node->bundle();
    $config = $this->configFactory->get('node_authlink.settings');
    $bundle_grants = ConfigurationHelper::getBundleGrants($config, $bundle);

    if (!ConfigurationHelper::isBundleEnabled($config, $bundle) || empty($bundle_grants)) {
      // Not possible to get here if the bundle has no authlinks enabled or
      // grants are empty. This is an urecoverable error, so triyng not to be
      // too dramatic.
      throw new NotFoundHttpException();
    }

    $form_builder = new NodeFormManager(
      $form,
      $this->entityTypeManager,
      $this->linkGenerator,
      $this->dateFormatter,
      $node,
      $bundle_grants);

    return $form_builder->build();
  }

  /**
   * Create authlink submit callback.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function createAuthlink(array &$form, FormStateInterface $form_state) {
    node_authlink_create($form_state->getBuildInfo()['args'][0]);
  }

  /**
   * Delete authlink submit callback.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function deleteAuthlink(array &$form, FormStateInterface $form_state) {
    node_authlink_delete($form_state->getBuildInfo()['args'][0]);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Display result.
    // @todo Can this be removed?
    foreach ($form_state->getValues() as $key => $value) {
      $this->messenger()->addMessage($key . ': ' . $value);
    }
  }

  /**
   * Custom access function for the node's authlinks form.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   * @param int $node Nid of the node being accessed.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   */
  public function access(AccountInterface $account, int $node) {

    $node_instance = Node::load($node);

    if ($node_instance instanceof NodeInterface) {
      $enable = $this->config('node_authlink.settings')->get('enable');
      if (isset($enable[$node_instance->bundle()]) && $enable[$node_instance->bundle()] && ($account->hasPermission('create and delete node authlinks') || $account->hasPermission(sprintf('create and delete node %s authlinks', $node_instance->bundle())))) {
        return AccessResult::allowed();
      }
    }

    return AccessResult::forbidden();
  }
}
