<?php

namespace Drupal\node_authlink\Form;

use Drupal\Core\Datetime\DateFormatter;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\LinkGenerator;
use Drupal\node\NodeInterface;

/**
 * Class to build $form structure for the form to manage a node's authlinks.
 */
class NodeFormManager {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Link generator service.
   *
   * @var \Drupal\Core\Utility\LinkGenerator
   */
  protected $linkGenerator;


  /**
   * Date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatter
   */
  protected $dateFormatter;

  /**
   * Array where the form is built.
   *
   * @var mixed[]
   */
  protected $form;

  /**
   * Entity storage for nodes.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $nodeStorage;

  /**
   * Node to build the form for.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node;

  /**
   * Grants of the content type of the node for which the form is being built.
   *
   * Grants for the bundle of this node as they are set in the
   * node_authlink.settings configuration entity: an array of operations that
   * the authlink should authorize. The key is the operation code and the
   * value is either 0 (not enabled) or the operation code as well (authlink
   * authorizes this operation).
   *
   * @var mixed[]
   */
  protected $bundleGrants;

  /**
   * List of enabled operations on the node.
   *
   * @var string[]
   *   Each has the operation machine name as key and value.
   */
  protected $enabledOps;

  /**
   * Is the view node operation enabled?
   *
   * @var bool
   */
  protected $viewOpEnabled;

  /**
   * Is the view revisions operaiton enabled?
   *
   * @var bool
   */
  protected $viewRevisionOpEnabled;

  /**
   * Information about the revisions.
   *
   * @var mixed[]
   *   Array with two keys:
   *     - current: information about the node's current revision.
   *     - rest: array with information about revisions that are not the current
   *         revision.
   *
   * @See NodeFormManager::addRevisionInfo() for the structure of the each
   * revision information.
   */
  protected $revisions;

  /**
   * Constructor function.
   *
   * @param array $form
   *   Initial $form structure. Form will be build from this.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager.
   * @param \Drupal\Core\Utility\LinkGenerator $link_generator
   *   Link Generator.
   * @param \Drupal\Core\Datetime\DateFormatter $date_formatter
   *   Date Formatter.
   * @param \Drupal\node\NodeInterface $node
   *   Node to build the form for.
   * @param $bundle_grants
   *   Node's bundle grants. See @see $this->bundleGrants.
   */
  public function __construct(
    array &$form,
    EntityTypeManagerInterface $entity_type_manager,
    LinkGenerator $link_generator,
    DateFormatter $date_formatter,
    NodeInterface $node,
    $bundle_grants) {

    $this->form = $form;
    $this->entityTypeManager = $entity_type_manager;
    $this->linkGenerator = $link_generator;
    $this->dateFormatter = $date_formatter;
    $this->node = $node;
    $this->bundleGrants = $bundle_grants;

    $this->nodeStorage = $this->entityTypeManager->getStorage('node');
  }

  /**
   * Main function that builds the form.
   *
   * @return array
   *   The built form.
   */
  public function build() {

    $this->enabledOps = array_filter($this->bundleGrants, fn($v) => $v !== 0);
    if (empty($this->enabledOps)) {
      $this->form['no_grants'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('This content type has no grants configured. Please edit the configuration of the content type and enable some grants to see the authlinks of this node.') . '</p>',
      ];
      return $this->form;
    }

    $this->form['disclaimer'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Use the following form to manage anonymous authlinks for performing View, View Revisions, Update or Delete tasks without any further authentication. The links available will depend on the configuration of this content type.') . '</p>',
    ];

    if (node_authlink_load_authkey($this->node->id())) {
      $this->buildFormForNodesWithAuthKey();
    }
    else {
      $this->buildFormForNodesWithNoAuthKey();
    }

    return $this->form;
  }

  /**
   * Builds the part of the form for nodes that have auth key already created.
   */
  public function buildFormForNodesWithAuthKey() {

    $this->form['delete'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete authlink'),
      '#weight' => 100,
      '#submit' => ['::deleteAuthlink'],
    ];

    $this->prepareViewOperations();

    foreach ($this->enabledOps as $op) {
      switch ($op) {
        case 'view':
          $this->addCurrentRevisionElem();
          break;

        case 'view revision':
          $this->addRevisionElems();
          break;

        case 'update':
          $this->addSimpleActionElem($this->t('Edit'), $op);
          break;

        case 'delete':
          $this->addSimpleActionElem($this->t('Delete'), $op);
          break;
      }
    }
  }

  /**
   * Prepares the form to hold data related to view operations.
   *
   * If there are view operations enabled (access to the current revision or
   * access to revisions), this method gathers info about the available
   * revisions and adds minimum form elements to display its data.
   */
  protected function prepareViewOperations() {
    $this->viewOpEnabled = in_array('view', $this->enabledOps);
    $this->viewRevisionOpEnabled = in_array('view revision', $this->enabledOps);

    if ($this->viewOpEnabled || $this->viewRevisionOpEnabled) {
      $this->prepareRevisionsData();
    }

    if ($this->viewRevisionOpEnabled) {
      $this->addRevisionTable();
    }

    if (array_intersect($this->enabledOps, ['view', 'update', 'delete'])) {
      $this->form['links'] = [
        '#type' => 'table',
        '#caption' => $this->t('Links'),
        '#header' => [
          $this->t('Action'),
          $this->t('Link'),
        ],
        '#rows' => [],
        '#weight' => 1,
      ];
    }
  }

  /**
   * Adds a form element for the current revision link.
   */
  protected function addCurrentRevisionElem() {
    $action_cell_content = $this->t('Current revision');

    if (!empty($this->revisions['current'])) {
      $current = $this->revisions['current'];
      $link_cell_content = $this->revisionInfoToRederElemnt($current, FALSE);
      $this->form['links']['current_revision'] = $this->getActionAndLinkElems($action_cell_content, $link_cell_content);
    }
  }

  /**
   * Adds a form element for the revision links.
   */
  protected function addRevisionElems() {
    if (!empty($this->revisions['rest'])) {
      $this->form['revisions']['rev'] = [
        'widget' => [
          '#type' => 'select',
          '#title' => $this->t('Select revision'),
          '#options' => [],
        ],
      ];

      $this->addRevisionLinkElems();
    }
  }

  /**
   * Adds revision links form elems and options on the select for revisions.
   *
   * Calculates the link info that should be displayed. All the revisions but
   * current revisions are displayed together. Using Drupal #states to only
   * display the one selected by the select elem.
   *
   * @See NodeFormBuilder::addRevisionInfo() for the structure of each
   * revision info ('current' and each item in 'rest').
   */
  public function addRevisionLinkElems() {
    $link_elems = [];
    if (!empty($this->revisions['rest'])) {
      foreach ($this->revisions['rest'] as $revision) {
        $link_elems[] = $this->revisionInfoToRederElemnt($revision);
        $this->form['revisions']['rev']['widget']['#options'][$revision['vid']] = $revision['text'];
      }
    }

    $this->form['revisions']['rev']['link'] = $link_elems;
  }

  /**
   * Adds a simple action form element.
   *
   * A simple action is either 'delete' or 'update'. Simple because are simple
   * to handler here.
   *
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $label
   *   Translated label for the operation.
   * @param string $op
   *   Machine name of the operation.
   */
  protected function addSimpleActionElem($label, $op) {
    $action_cell_content = $label;
    $url = node_authlink_get_url($this->node, $op);

    if (empty($url)){
      return;
    }

    $link_cell_content = [
      '#type' => 'item',
      '#markup' => $this->urlToLinkMarkup($url),
    ];

    $this->form['links'][$op] = $this->getActionAndLinkElems($action_cell_content, $link_cell_content);
  }

  /**
   * Fetches and prepares data required to handle the revisions.
   */
  public function prepareRevisionsData() {

    $revision_dataset = $this->queryRevisions();

    if (!empty($revision_dataset) && is_array($revision_dataset)) {
      $this->revisions = [
        'current' => NULL,
        'rest' => [],
      ];

      foreach (array_keys($revision_dataset) as $vid) {
        $this->addRevisionInfo($vid);
      }
    }
  }

  /**
   * Adds a table form element to hold the revision displayed information.
   */
  protected function addRevisionTable() {

    $this->form['revisions'] = [
      '#type' => 'table',
      '#caption' => $this->t('Revision links'),
      '#weight' => 50,
    ];

    if (!empty($this->revisions['rest'])) {
      $this->form['revisions']['#header'] = [
        $this->t('Revision'),
        $this->t('Link'),
      ];
    }
    else {
      $this->form['revisions']['#header'] = [$this->t('<p>No revisions available for this node.</p>')];
    }
  }

  /**
   * Returns a pair of form elements used to display action and link(s).
   *
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $action
   *   Sanitized and translated markup with the action label.
   * @param array $link
   *   A render array with the link o links to display.
   *
   * @return array
   *   A render array with the requested elems.
   */
  public function getActionAndLinkElems($action, $link) {
    return [
      'action ' => [
        '#type' => 'markup',
        '#markup' => $action,
      ],
      'links' => $link,
    ];
  }

  /**
   * Returns a Drupal's render element from a revision info.
   *
   * @param array $revision
   *   Revision info to turn into a render element.
   *   @See NodeFormBuilder::addRevisionInfo() for the structure of each
   *   revision info ('current' and each item in 'rest').
   *
   * @return array
   *   A render element (a Drupal's Render Array element).
   */
  protected function revisionInfoToRederElemnt($revision, $hideable = TRUE) {

    $link = $this->urlToLinkMarkup($revision['url']);

    $elem = [
      '#type' => 'item',
      '#markup' => $link,
    ];

    if ($hideable) {
      $elem['#states'] = [
        'visible' => [
          '[name="revisions[rev][widget]"]' => ['value' => $revision['vid']],
        ],
      ];
    }

    return $elem;
  }

  /**
   * Queries the database for the revisions of the node the form is built for.
   *
   * @param int $max
   *   Max number of revisions to retrieve.
   *
   * @return array
   *   Array of vid => nid. Because we are querying a unique node all the nids
   *   will be the same value.
   */
  protected function queryRevisions(int $max = 50) {
    $entity_type = $this->node->getEntityType();

    $entity_id_column = $entity_type->getKey('id');
    $entity_revision_column = $entity_type->getKey('revision');

    if (!$entity_id_column  || !$entity_revision_column ) {
      return [];
    }

    return $this->nodeStorage->getQuery()
      ->allRevisions()
      ->condition($entity_id_column, $this->node->id())
      ->sort($entity_revision_column, 'DESC')
      ->range(0, $max)
      ->accessCheck(TRUE)
      ->execute();
  }

  /**
   * Adds required info from a revision to the form so it can be displayed.
   *
   * Each revision information is an array with the folowing keys:
   *   - vid: revision id.
   *   - text: human readable label.
   *   - url: \Drupal\Core\GeneratedUrl with the revision's url.
   *
   * @param int $vid
   */
  protected function addRevisionInfo($vid) {
    /** @var \Drupal\node\Entity\Node|null $revision */
    $revision = $this->nodeStorage->loadRevision($vid);

    if (empty($revision)) {
      return;
    }

    $langcode = $this->node->language()->getId();

    // Only show revisions that are affected by the language that is being
    // displayed.
    if ($revision->hasTranslation($langcode) && $revision->getTranslation($langcode)->isRevisionTranslationAffected()) {

      if ($revision->isDefaultRevision() && $this->viewOpEnabled) {
        $this->revisions['current'] = [
          'vid' => $vid,
          'text' => $this->t('Current revision'),
          'url' => node_authlink_get_url($this->node),
        ];
      }
      elseif ($this->viewRevisionOpEnabled) {
        $this->revisions['rest'][] = [
          'vid' => $vid,
          'text' => $this->dateFormatter->format($revision->revision_timestamp->value, 'short') . " (rev: $vid)",
          'url' => node_authlink_get_url($this->node, 'view', $vid),
        ];
      }
    }
  }

  /**
   * Builds the part of the form for nodes that have no auth key yet.
   *
   * This means only the button to generate the keys.
   */
  public function buildFormForNodesWithNoAuthKey() {

    $this->form['no_auhtlink_notice'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('No authlink has been created for this node yet.') . '</p>',
    ];

    $this->form['create'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create authlink'),
      '#weight' => 100,
      '#submit' => ['::createAuthlink'],
    ];
  }

  /**
   * Returns markup for a given URL.
   *
   * The link text is the URL itself.
   *
   * @param \Drupal\Core\Url $url
   *   URL to generate markup from.
   *
   * @return string
   *   Generated markup.
   */
  protected function urlToLinkMarkup($url) {
    return $this->linkGenerator->generate($url->toString(), $url);
  }

}
