<?php

namespace Drupal\node_authlink\Plugin\views\field;

use Drupal\Core\Url;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;

/**
 * A handler to provide a link to execute a function.
 *
 * @ViewsField("authlink_delete_link")
 */
class AuthlinkDeleteLink extends FieldPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    AccountInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Define the available options
   *
   * @return array
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $entity = $values->_relationship_entities['nid'] ?? NULL;
    if ($entity instanceof NodeInterface && $this->hasDeleteAuthlinkPermission($entity)) {
      $url = Url::fromRoute('node_authlink.delete_form', [
        'node' => $entity->id(),
      ]);

      return [
        '#type' => 'link',
        '#title' => $this->t('Delete Authlink'),
        '#url' => $url,
        '#attributes' => [
          'class' => ['button', 'button--small', 'button--danger'],
        ],
      ];
    }
    else {
      return '';
    }
  }

  /**
   * Checks if the user has permission to delete the authlink.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node object.
   *
   * @return bool
   *   TRUE if the user has permission, FALSE otherwise.
   */
  protected function hasDeleteAuthlinkPermission(NodeInterface $entity) {
    return $this->currentUser->hasPermission('create and delete node authlinks') ||
      $this->currentUser->hasPermission(sprintf('create and delete node %s authlinks', $entity->bundle()));
  }

}
