<?php

declare(strict_types=1);

namespace Drupal\Tests\node_form_overrides\Functional;

/**
 * Tests node form overrides without token module.
 *
 * @group node_form_overrides
 */
class NodeFormOverridesTest extends NodeFormOverridesTestBase {

  /**
   * Tests that create button label is overridden.
   */
  public function testCreateButtonLabel(): void {
    $this->setOverrideSettings([
      'insert_button' => 'Publish Article',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/add/test_content');

    $this->assertSession()->buttonExists('Publish Article');
  }

  /**
   * Tests that update button label is overridden.
   */
  public function testUpdateButtonLabel(): void {
    $this->setOverrideSettings([
      'update_button' => 'Save Changes',
    ]);

    // Create a node first.
    $node = $this->drupalCreateNode(['type' => 'test_content']);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    $this->assertSession()->buttonExists('Save Changes');
  }

  /**
   * Tests that create page title is overridden.
   */
  public function testCreatePageTitle(): void {
    $this->setOverrideSettings([
      'insert_title' => 'Create New Article',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/add/test_content');

    $this->assertSession()->titleEquals('Create New Article | Drupal');
  }

  /**
   * Tests that update page title is overridden.
   */
  public function testUpdatePageTitle(): void {
    $this->setOverrideSettings([
      'update_title' => 'Edit Article',
    ]);

    // Create a node first.
    $node = $this->drupalCreateNode(['type' => 'test_content']);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    $this->assertSession()->titleEquals('Edit Article | Drupal');
  }

  /**
   * Tests that delete form title is overridden.
   */
  public function testDeleteFormTitle(): void {
    $this->setOverrideSettings([
      'delete_form_title' => 'Delete this article?',
    ]);

    // Create a node first.
    $node = $this->drupalCreateNode(['type' => 'test_content']);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/delete');

    $this->assertSession()->titleEquals('Delete this article? | Drupal');
  }

  /**
   * Tests that delete form description is overridden.
   */
  public function testDeleteFormDescription(): void {
    $this->setOverrideSettings([
      'delete_form_description' => 'Warning: This will permanently remove the article.',
    ]);

    // Create a node first.
    $node = $this->drupalCreateNode(['type' => 'test_content']);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/delete');

    $this->assertSession()->pageTextContains('Warning: This will permanently remove the article.');
  }

  /**
   * Tests that default values work when no overrides are set.
   */
  public function testDefaultValues(): void {
    // Don't set any overrides.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/add/test_content');

    // Should have default Drupal button.
    $this->assertSession()->buttonExists('Save');
  }

  /**
   * Tests that empty override values use defaults.
   */
  public function testEmptyOverridesUseDefaults(): void {
    $this->setOverrideSettings([
      'insert_button' => '',
      'insert_title' => '',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/add/test_content');

    // Empty values should not override - getThirdPartySetting returns empty
    // string which is falsy, so no override is applied.
    $this->assertSession()->buttonExists('Save');
  }

  /**
   * Tests the node type settings form contains override fields.
   */
  public function testNodeTypeSettingsForm(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/types/manage/test_content');

    // Check that the override fields exist.
    $this->assertSession()->fieldExists('node_form_overrides[insert_button]');
    $this->assertSession()->fieldExists('node_form_overrides[update_button]');
    $this->assertSession()->fieldExists('node_form_overrides[insert_title]');
    $this->assertSession()->fieldExists('node_form_overrides[update_title]');
    $this->assertSession()->fieldExists('node_form_overrides[delete_form_title]');
    $this->assertSession()->fieldExists('node_form_overrides[delete_form_description]');
  }

  /**
   * Tests saving override settings via the node type form.
   */
  public function testSaveOverrideSettings(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/types/manage/test_content');

    $edit = [
      'node_form_overrides[insert_button]' => 'Create Test',
      'node_form_overrides[update_button]' => 'Update Test',
      'node_form_overrides[insert_title]' => 'New Test Content',
      'node_form_overrides[update_title]' => 'Edit Test Content',
      'node_form_overrides[delete_form_title]' => 'Delete Test?',
      'node_form_overrides[delete_form_description]' => 'This is permanent.',
    ];
    $this->submitForm($edit, 'Save');

    // Verify settings were saved by checking the node add form.
    $this->drupalGet('node/add/test_content');
    $this->assertSession()->buttonExists('Create Test');
    $this->assertSession()->titleEquals('New Test Content | Drupal');
  }

}
