<?php

declare(strict_types=1);

namespace Drupal\Tests\node_form_overrides\Functional;

/**
 * Tests node form overrides with token module enabled.
 *
 * @group node_form_overrides
 */
class NodeFormOverridesTokenTest extends NodeFormOverridesTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'node_form_overrides',
    'token',
  ];

  /**
   * Tests token replacement in create page title.
   */
  public function testCreatePageTitleWithToken(): void {
    $this->setOverrideSettings([
      'insert_title' => 'Add new [node:content-type:name]',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/add/test_content');

    // Token should be replaced with the content type name.
    $this->assertSession()->titleEquals('Add new Test Content | Drupal');
  }

  /**
   * Tests token replacement in update page title.
   */
  public function testUpdatePageTitleWithToken(): void {
    $this->setOverrideSettings([
      'update_title' => 'Editing: [node:title]',
    ]);

    // Create a node with a specific title.
    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => 'My Test Article',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Token should be replaced with the node title.
    $this->assertSession()->titleEquals('Editing: My Test Article | Drupal');
  }

  /**
   * Tests token replacement in delete form title.
   */
  public function testDeleteFormTitleWithToken(): void {
    $this->setOverrideSettings([
      'delete_form_title' => 'Delete "[node:title]"?',
    ]);

    // Create a node with a specific title.
    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => 'Article to Delete',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/delete');

    // Token should be replaced with the node title.
    $this->assertSession()->titleEquals('Delete "Article to Delete"? | Drupal');
  }

  /**
   * Tests token replacement in delete form description.
   */
  public function testDeleteFormDescriptionWithToken(): void {
    $this->setOverrideSettings([
      'delete_form_description' => 'You are about to delete "[node:title]". This cannot be undone.',
    ]);

    // Create a node with a specific title.
    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => 'Important Article',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/delete');

    // Token should be replaced in the description.
    $this->assertSession()->pageTextContains('You are about to delete "Important Article". This cannot be undone.');
  }

  /**
   * Tests multiple tokens in a single title.
   */
  public function testMultipleTokensInTitle(): void {
    $this->setOverrideSettings([
      'update_title' => 'Edit [node:content-type:name]: [node:title]',
    ]);

    // Create a node with a specific title.
    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => 'Sample Post',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Both tokens should be replaced.
    $this->assertSession()->titleEquals('Edit Test Content: Sample Post | Drupal');
  }

  /**
   * Tests that token browser link appears in node type form.
   */
  public function testTokenBrowserDisplayed(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/types/manage/test_content');

    // The token browser link should be present when token module is enabled.
    $this->assertSession()->linkExists('Browse available tokens.');
  }

  /**
   * Tests that unreplaced tokens are left as-is for new nodes.
   *
   * Some tokens like [node:nid] won't have values for new nodes.
   */
  public function testUnreplacedTokensOnNewNode(): void {
    $this->setOverrideSettings([
      'insert_title' => 'New [node:content-type:name] (ID: [node:nid])',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/add/test_content');

    // Content type token should be replaced, but nid won't exist yet.
    // Token module leaves unreplaced tokens as-is by default.
    $this->assertSession()->titleEquals('New Test Content (ID: ) | Drupal');
  }

  /**
   * Tests that special HTML characters in node titles are displayed correctly.
   *
   * Regression test for issue where & was displayed as &amp; in page titles.
   *
   * @see https://www.drupal.org/project/node_form_overrides/issues/XXXXXXX
   */
  public function testSpecialCharactersInTitle(): void {
    $this->setOverrideSettings([
      'update_title' => 'Edit: [node:title]',
    ]);

    // Create a node with special characters in the title.
    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => 'Tom & Jerry',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // The & should NOT be double-encoded as &amp;
    // The page title should display correctly.
    $this->assertSession()->titleEquals('Edit: Tom & Jerry | Drupal');

    // Also verify it's not showing the encoded version.
    $this->assertSession()->responseNotContains('Tom &amp;amp; Jerry');
  }

  /**
   * Tests special characters in delete form title.
   */
  public function testSpecialCharactersInDeleteTitle(): void {
    $this->setOverrideSettings([
      'delete_form_title' => 'Delete "[node:title]"?',
    ]);

    // Create a node with special characters.
    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => 'Rock & Roll',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/delete');

    // The & should display correctly.
    $this->assertSession()->titleEquals('Delete "Rock & Roll"? | Drupal');
  }

  /**
   * Tests special characters in delete form description.
   */
  public function testSpecialCharactersInDeleteDescription(): void {
    $this->setOverrideSettings([
      'delete_form_description' => 'Deleting "[node:title]" is permanent.',
    ]);

    // Create a node with special characters.
    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => 'Salt & Pepper',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/delete');

    // The & should display correctly in the description.
    $this->assertSession()->pageTextContains('Deleting "Salt & Pepper" is permanent.');
  }

  /**
   * Tests that HTML in node titles is escaped in page titles.
   *
   * Ensures XSS prevention - script tags should be escaped, not executed.
   */
  public function testHtmlEscapedInTitle(): void {
    $this->setOverrideSettings([
      'update_title' => 'Edit: [node:title]',
    ]);

    // Create a node with HTML/script in the title.
    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => '<script>alert("xss")</script>',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // The script tag should be escaped, not present as raw HTML.
    // Page title is escaped by Drupal, so we check it doesn't execute.
    $this->assertSession()->responseNotContains('<script>alert("xss")</script>');
    // The escaped version should be in the page source.
    $this->assertSession()->responseContains('&lt;script&gt;');
  }

  /**
   * Tests that HTML in node titles is escaped in delete form title.
   */
  public function testHtmlEscapedInDeleteTitle(): void {
    $this->setOverrideSettings([
      'delete_form_title' => 'Delete [node:title]?',
    ]);

    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => '<img src=x onerror=alert("xss")>',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/delete');

    // Raw HTML should not be present - should be escaped.
    $this->assertSession()->responseNotContains('<img src=x onerror=alert("xss")>');
    $this->assertSession()->responseContains('&lt;img');
  }

  /**
   * Tests that HTML in tokens is handled safely in description markup.
   *
   * The description uses #markup which goes through Xss::filter().
   * Token values are pre-escaped by replace(), providing defense in depth.
   */
  public function testHtmlInDeleteDescription(): void {
    $this->setOverrideSettings([
      'delete_form_description' => 'Deleting "[node:title]" cannot be undone.',
    ]);

    $node = $this->drupalCreateNode([
      'type' => 'test_content',
      'title' => '<script>alert("xss")</script>',
    ]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('node/' . $node->id() . '/delete');

    // Script tag should not be executable - should be escaped or stripped.
    $this->assertSession()->responseNotContains('<script>alert("xss")</script>');
    // The escaped form should appear (token values are HTML-escaped).
    $this->assertSession()->pageTextContains('<script>alert("xss")</script>');
  }

}
