<?php

namespace Drupal\node_health\Commands;

use Drush\Commands\DrushCommands;

/**
 * Drush commands for node revision cleanup.
 */
class NodeHealthCommands extends DrushCommands {

  /**
   * Deletes old revisions of all nodes, keeping the latest N revision.
   *
   * @command node-health:clean-up-revision
   * @aliases nrh:cur
   *
   * @usage node-health:clean-up-revision min --content-type=node --nids=1,2,3,4
   */
  public function NodeHealthCleanUpRevision(int $revisions_to_keep = 1, array $options = ['content-type' => NULL, 'nids' => NULL]) {
    if ($revisions_to_keep < 1) {
      $this->io()->error('The number of revisions to keep must be at least 1.');
      return;
    }

    $content_type = $options['content-type'];
    $nids = $options['nids'];

    $this->output()->writeln('----------------------------------------');
    $this->output()->writeln("Starting node revision cleanup.");
    $this->output()->writeln("Keeping the latest {$revisions_to_keep} revisions per node.");
    if ($content_type) {
      $this->output()->writeln("Filtering by content type: {$content_type}");
    }
    $this->output()->writeln('----------------------------------------');

    $query = \Drupal::entityQuery('node')
      ->accessCheck(FALSE);

    if ($content_type) {
      $query->condition('type', $content_type);
    }

    if ($nids) {
      $query->condition('nid', explode(",", $nids), 'IN');
    }

    $nids = $query->execute();

    if (empty($nids)) {
      $this->output()->writeln('No nodes found for the given criteria.');
      return;
    }

    $nids = array_reverse($nids);

    $total = count($nids);
    $count = 0;

    foreach ($nids as $nid) {
      $count++;
      $this->output()->writeln("[$count/$total] Processing node ID {$nid}...");
      $this->processNodeRevisions($nid, $revisions_to_keep);
    }

    $this->output()->writeln('✔ Node revision cleanup completed.');
  }

  /**
   * Deletes old revisions of all nodes, keeping the latest N revision.
   *
   * @command node-health:clean-up-revision-random
   * @aliases nrh:curr
   *
   * @usage node-health:clean-up-revision-random min --content-type=node --nids=1,2,3,4
   */
  public function NodeHealthCleanUpRevisionRandom(int $revisions_to_keep = 1, array $options = ['content-type' => NULL, 'limit' => 1000]) {
    if ($revisions_to_keep < 1) {
      $this->io()->error('The number of revisions to keep must be at least 1.');
      return;
    }

    $content_type = $options['content-type'];
    $limit = $options['limit'] ?? 1000;

    $this->output()->writeln('----------------------------------------');
    $this->output()->writeln("Starting node revision cleanup.");
    $this->output()->writeln("Keeping the latest {$revisions_to_keep} revisions per node.");
    if ($content_type) {
      $this->output()->writeln("Filtering by content type: {$content_type}");
    }
    $this->output()->writeln('----------------------------------------');

    $query = \Drupal::database()->select('node_field_data', 'n');
    $query->addField('n', 'nid');
    if ($content_type) {
      $query->condition('n.type', $content_type);
    }
    $query->addExpression('RAND()', 'random_order');
    $query->orderBy('random_order');
    $query->range(0, $limit);

    $nids = $query->execute()->fetchCol();

    if (empty($nids)) {
      $this->output()->writeln('No nodes found for the given criteria.');
      return;
    }

    $nids = array_reverse($nids);

    $total = count($nids);
    $count = 0;

    foreach ($nids as $nid) {
      $count++;
      $this->output()->writeln("[$count/$total] Processing node ID {$nid}...");
      $this->processNodeRevisions($nid, $revisions_to_keep);
    }

    $this->output()->writeln('✔ Node revision cleanup completed.');
  }

  /**
   * Process revisions for a single node.
   */
  protected function processNodeRevisions(int $nid, int $revisions_to_keep): void {
    $storage = \Drupal::entityTypeManager()->getStorage('node');
    $node = $storage->load($nid);
    $default_revision_id = $node->getRevisionId();

    if (!$node) {
      $this->output()->writeln("Node with ID {$nid} not found.");
      return;
    }

    $revision_ids = $storage->revisionIds($node);

    $revision_ids = array_combine($revision_ids, $revision_ids);
    unset($revision_ids[$default_revision_id]);

    if (!$revision_ids || count($revision_ids) <= $revisions_to_keep) {
      $this->output()->writeln("Node {$nid} has " . ($revision_ids ? count($revision_ids) : 0) . " revisions. Nothing to delete.");
      return;
    }

    rsort($revision_ids);

    if (count($revision_ids) > $revisions_to_keep) {
      // Keep only the first $revisions_to_keep, removing from the end.
      $revision_ids = array_slice($revision_ids, $revisions_to_keep);
    }

    foreach ($revision_ids as $revision_id) {
      try {
        $storage->deleteRevision($revision_id);
        $this->output()->writeln(" - Deleted revision {$revision_id} of node {$nid}.");
        \Drupal::logger('node_health')->notice('Deleted revision @rid of node @nid', [
          '@rid' => $revision_id,
          '@nid' => $nid,
        ]);
      }
      catch (\Exception $e) {
        $this->output()->writeln(" - ❌ Failed to delete revision {$revision_id}: {$e->getMessage()}");
        \Drupal::logger('node_health')->error('Failed to delete revision @rid: @msg', [
          '@rid' => $revision_id,
          '@msg' => $e->getMessage(),
        ]);
      }
    }
  }

}
