<?php

namespace Drupal\node_health\Commands;

use Drupal\node\Entity\Node;
use Drush\Commands\DrushCommands;

/**
 * A Drush commandfile.
 *
 * In addition to this file, you need a drush.services.yml
 * in root of your module, and a composer.json file that provides the name
 * of the services file to use.
 *
 * See these files for an example of injecting Drupal services:
 *   - http://cgit.drupalcode.org/devel/tree/src/Commands/DevelCommands.php
 *   - http://cgit.drupalcode.org/devel/tree/drush.services.yml
 */
class NodeHealthNodes extends DrushCommands {

  /**
   * Make a batch structure.
   */
  public static function makeBatch($content_type, $size) {
    $operations = [];
    $query = \Drupal::entityQuery('node')->accessCheck(FALSE);

    if (!empty($content_type)) {
      $content_type = explode(',', $content_type);
      $content_type = array_filter($content_type);
      $query->condition('type', $content_type, "IN");
    }
    $query = $query->sort('nid', 'ASC');
    $nids = $query->execute();

    $nids_buffer = [];
    $i = 0;
    foreach ($nids as $key => $nid) {
      $nids_buffer[] = $nid;
      if (($i + 1) % $size == 0) {
        $operations[] = [
          '\Drupal\node_health\Commands\NodeHealthNodes::update',
          [$nids_buffer],
        ];
        $nids_buffer = [];
      }
      $i++;
    }

    if (count($nids_buffer)) {
      $operations[] = [
        '\Drupal\node_health\Commands\NodeHealthNodes::update',
        [$nids_buffer],
      ];
    }

    if (!empty($operations)) {
      $batch = [
        'title' => t('Starting updating...'),
        'operations' => $operations,
        'finished' => '\Drupal\node_health\Commands\NodeHealthNodes::NodeHealthNodesFinishedCallback',
      ];
      return $batch;
    }

    return FALSE;
  }

  /**
   * Delete nodes.
   *
   * @param string $type
   *   The required type parameter.
   *
   * @command NodeHealthNodes
   * @aliases nrh:dn
   */
  public function NodeHealthNodes($type = NULL) {
    $this->logger()->notice("Start sync tags");
    $this->output()->writeln("Preparing batch");

    $size = 100;

    $batch = self::makeBatch($type, $size);
    if ($batch) {
      batch_set($batch);
      drush_backend_batch_process();
      $batch = &batch_get();
      $this->logger()->notice("Finishing batch.");
    }
    else {
      $this->output()->writeln("No node found to run the batch.");
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function update($nids, &$context) {
    foreach ($nids as $key => $nid) {
      $node = Node::load($nid);
      if ($node instanceof Node) {
        $node->delete();
      }
    }
    $context['message'] = t(
      'Updatted: @nids',
      ['@nids' => count($context['results'])]
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function NodeHealthNodesFinishedCallback($success, $results, $operations) {
    if ($success) {
      $message = \Drupal::translation()->formatPlural(
        count($results),
        'One node processed.',
        '@count recipes processed.'
      );
    }
    else {
      $message = t('Finished with an error.');
    }
    \Drupal::messenger()->addMessage($message);
  }

}
