<?php

namespace Drupal\node_health\Commands;

use Drupal\field\Entity\FieldStorageConfig;
use Drush\Commands\DrushCommands;

/**
 * Commands to clean up orphaned node_r__* tables.
 */
class NodeHealthOrphanRTablesCommands extends DrushCommands {

  /**
   * Clean up orphaned node_r__* tables.
   *
   * @command node_health:clean-orphan-revision-tables
   * @aliases nh-clean-r
   */
  public function cleanOrphanRevisionTables() {
    $schema = \Drupal::database()->getConnectionOptions()['database'];

    // Find all node_r__* tables.
    $all_tables = \Drupal::database()->query("
      SELECT table_name FROM information_schema.tables
      WHERE table_schema = :schema
        AND table_name LIKE 'node_r\\_\\_%'
    ", [':schema' => $schema])->fetchCol();

    // Get all node field storage definitions.
    $field_storages = FieldStorageConfig::loadMultiple();
    $used_tables = [];
    $hash_to_field = [];

    foreach ($field_storages as $field_storage) {
      if ($field_storage->getTargetEntityTypeId() !== 'node') {
        continue;
      }

      $field_name = $field_storage->getName();
      $hash = substr(hash('sha256', $field_storage->getUniqueStorageIdentifier() ?? ''), 0, 10);
      $hash_to_field["node_r__{$hash}"] = $field_name;
      $used_tables[$field_name] = 'node_r__' . $hash;
    }

    $orphaned = array_diff($all_tables, $used_tables);

    if (empty($orphaned)) {
      $this->output()->writeln("✅ No orphaned node_r__* tables found.");
      return;
    }

    $this->output()->writeln("🚨 Found " . count($unused) . " orphaned tables:");
    foreach ($orphaned as $table) {
      $this->output()->writeln(" - $table");
    }

    if ($this->io()->confirm('Delete these orphaned tables now?')) {
      foreach ($orphaned as $table) {
        \Drupal::database()->query("DROP TABLE `$table`");
        $this->output()->writeln("🗑️ Dropped: $table");
      }
    }
    else {
      $this->output()->writeln("❌ No tables were dropped.");
    }
  }

}
