<?php

namespace Drupal\node_health\Commands;

use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for processing queues.
 */
class NodeHealthQueueProcessorCommands extends DrushCommands {

  protected $queueFactory;
  protected $queueWorkerManager;

  public function __construct(QueueFactory $queue_factory, QueueWorkerManagerInterface $queue_worker_manager) {
    parent::__construct();
    $this->queueFactory = $queue_factory;
    $this->queueWorkerManager = $queue_worker_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('queue'),
      $container->get('plugin.manager.queue_worker')
    );
  }

  /**
   * Process a queue with progress output.
   *
   * @command node-health-queue-processor:run
   * @param string $queue_name
   *   The queue to process.
   *
   * @usage drush node-health-queue-processor:run entity_reference_revisions_orphan_purger
   * @usage drush node-health-queue-processor:run node_revision_delete
   */
  public function process($queue_name) {
    $queue = $this->queueFactory->get($queue_name);
    $total = $queue->numberOfItems();

    if ($total === 0) {
      $this->output()->writeln("Queue '$queue_name' has no items.");
      return;
    }

    $this->output()->writeln("Processing $total items from queue '$queue_name'...");

    $worker = $this->queueWorkerManager->createInstance($queue_name);
    $processed = 0;

    while ($item = $queue->claimItem()) {
      try {
        $worker->processItem($item->data);
        $queue->deleteItem($item);
        $processed++;
        $this->output()->writeln("Processed: $processed / $total");
      }
      catch (\Exception $e) {
        $this->logger()->error("Error processing item: " . $e->getMessage());
      }
    }

    $this->output()->writeln("Finished processing $processed items from '$queue_name'.");
  }

}
