<?php

namespace Drupal\node_health\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\node_health\Helper\NodeHealthFormatSize;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for listing node health chart reports.
 */
class NodeHealthChartListController extends ControllerBase {

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Inject the database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * Build the report table.
   */
  public function report() {
    $header = [
      'table_name' => $this->t('Table Name'),
      'init_size' => $this->t('Init Size'),
      'init_rows' => $this->t('Init Rows'),
      'init_timestamp' => $this->t('Init Timestamp'),
      'current_size' => $this->t('Last Size'),
      'current_rows' => $this->t('Last Rows'),
      'current_timestamp' => $this->t('Last Timestamp'),
      'chart' => $this->t('Chart'),
    ];

    // Get unique table names.
    $query = \Drupal::database()->select('node_health_sizes', 's');
    $query->innerJoin('node_health_table_names', 't', 's.table_id = t.id');
    // $query->fields('s', ['size', 'timestamp']);
    $query->fields('t', ['table_name']);
    $query->orderBy('s.timestamp', 'DESC');

    $result = $query->execute()->fetchCol();

    $result = array_unique($result);

    $rows = [];

    foreach ($result as $table_name) {
      // Get the init size and timestamp (earliest).
      $init_query = \Drupal::database()->select('node_health_sizes', 's');
      $init_query->innerJoin('node_health_table_names', 't', 's.table_id = t.id');
      $init_query->fields('s', ['size', 'timestamp', 'count']);
      $init_query->fields('t', ['table_name']);
      $init_query->condition('t.table_name', $table_name);
      $init_query->orderBy('s.timestamp', 'ASC');
      $init_query->range(0, 1);
      $init_result = $init_query->execute()->fetchAssoc();

      // Get the current size and timestamp (latest).
      $current_query = \Drupal::database()->select('node_health_sizes', 's');
      $current_query->innerJoin('node_health_table_names', 't', 's.table_id = t.id');
      $current_query->fields('s', ['size', 'timestamp', 'count']);
      $current_query->fields('t', ['table_name']);
      $current_query->orderBy('s.timestamp', 'DESC');
      $current_query->range(0, 1);
      $current_query->condition('t.table_name', $table_name);
      $current_result = $current_query->execute()->fetchAssoc();

      // Build the chart link.
      $url = Url::fromUri('internal:/admin/reports/node-health/chart/' . $table_name);
      $link = Link::fromTextAndUrl($this->t('View Chart 📊'), $url)->toString();

      $rows[] = [
        'table_name' => $table_name,
        'init_size' => $init_result ? $init_result['size'] : 0,
        'init_rows' => $init_result ? $init_result['count'] : 0,
        'init_timestamp' => $init_result ? date('Y-m-d H:i:s', $init_result['timestamp']) : '-',
        'current_size' => $current_result ? $current_result['size'] : 0,
        'current_rows' => $current_result ? $current_result['count'] : 0,
        'current_timestamp' => $current_result ? date('Y-m-d H:i:s', $current_result['timestamp']) : '-',
        'chart' => ($link),
      ];
    }

    // Sort by 'size' in descending order.
    usort($rows, function ($a, $b) {
      return $b['current_size'] <=> $a['current_size'];
    });

    foreach ($rows as $key => $value) {
      $rows[$key]['init_size'] = NodeHealthFormatSize::formatSize($value['init_size']);
      $rows[$key]['init_rows'] = number_format($value['init_rows'] ?? 0);
      $rows[$key]['current_size'] = NodeHealthFormatSize::formatSize($value['current_size']);
      $rows[$key]['current_rows'] = number_format($value['current_rows'] ?? 0);
    }

    return [
      'fieldset' => [
        '#type' => 'fieldset',
        '#title' => $this->t('Node Revision Health Size Tables'),
        '#attributes' => [
          'class' => ['node-health-chart-list'],
        ],
        '#title' => $this->t('Node Revision Health Size Tables'),
        'description' => [
          '#markup' => '<p>' . $this->t('This table shows the results collected by the cron.') . '</p>',
        ],
        'table' => [
          '#type' => 'table',
          '#header' => $header,
          '#rows' => $rows,
          '#empty' => $this->t('No data found.'),
        ],
      ],
    ];

  }

}
