<?php

namespace Drupal\node_health\Controller;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for Node Health revision reports and batch operations.
 */
class NodeHealthReportController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a NodeHealthReportController object.
   */
  public function __construct(Connection $database, RequestStack $request_stack, EntityTypeManagerInterface $entity_type_manager, MessengerInterface $messenger) {
    $this->database = $database;
    $this->requestStack = $request_stack;
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('messenger')
    );
  }

  /**
   * Displays the Node revisions report with filters and pagination.
   *
   * @return array
   *   Render array for the report.
   */
  public function report() {
    $request = $this->requestStack->getCurrentRequest();
    $min_revisions = (int) $request->query->get('min', 0);
    $nid = $request->query->get('nid', '');
    $title = $request->query->get('title', '');
    $sort = $request->query->get('sort', 'desc');
    $content_type_filter = $request->query->get('type');
    $items_per_page = (int) $request->query->get('items_per_page') ?? 50;
    $current_page = (int) $request->query->get('page') ?? 1;
    $items_per_page = in_array($items_per_page, [10, 20, 50, 100, 500, 1000]) ? $items_per_page : 50;

    $content_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $options = ['' => $this->t('- All types -')];
    foreach ($content_types as $machine => $type) {
      $options[$machine] = $type->label();
    }

    $filters = [
      '#type' => 'fieldset',
      '#title' => $this->t('Filters'),
      'form' => [
        '#type' => 'container',
        '#attributes' => ['id' => ['node-health-form']],
        'form' => [
          '#type' => 'container',
          'title' => [
            '#type' => 'textfield',
            '#title' => $this->t('Title'),
            '#size' => 25,
            '#value' => $title,
            '#attributes' => ['name' => 'title'],
          ],
          'nid' => [
            '#type' => 'textfield',
            '#title' => $this->t('Nid'),
            '#value' => $nid,
            '#size' => 25,
            '#attributes' => ['name' => 'nid'],
          ],
          'content_type' => [
            '#type' => 'select',
            '#title' => $this->t('Content type'),
            '#options' => $options,
            '#value' => $content_type_filter,
            '#attributes' => ['name' => 'type'],
          ],
          'min' => [
            '#type' => 'number',
            '#title' => $this->t('Revision Count'),
            '#size' => 25,
            '#value' => $min_revisions,
            '#attributes' => ['name' => 'min'],
          ],
          'items_per_page' => [
            '#type' => 'select',
            '#title' => $this->t('Items per page'),
            '#options' => [10 => 10, 20 => 20, 50 => 50, 100 => 100, 500 => 500, 1000 => 1000],
            '#default_value' => $items_per_page,
            '#value' => $items_per_page,
            '#attributes' => ['name' => 'items_per_page'],
          ],
          'submit' => [
            '#type' => 'submit',
            '#value' => $this->t('Filter'),
            '#attributes' => ['type' => 'submit'],
          ],
        ],
      ],
    ];

    $header = [
      'nid' => $this->t('NID'),
      'title' => $this->t('Title'),
      'type' => $this->t('Content Type'),
      'revision_count' => [
        'data' => $this->t('Revision Count'),
        'field' => 'revision_count',
        'specifier' => 'revision_count',
      ],
      'operations' => $this->t('Operations'),
    ];

    // Create sortable links.
    $sort_link = Url::fromRoute('<current>', ['min' => $min_revisions], [
      'query' => ['sort' => $sort === 'desc' ? 'asc' : 'desc'],
    ])->toString();
    $header['revision_count']['data'] = Markup::create('<a href="' . $sort_link . '">' . $this->t('Revision Count') . '</a>');

    // Base query with COUNT aggregation.
    $query = $this->database->select('node_field_data', 'n');
    $query->join('node_revision', 'r', 'n.nid = r.nid');
    $query->addExpression('COUNT(r.vid)', 'revision_count');
    $query->fields('n', ['nid', 'title', 'type', 'langcode']);
    $query->groupBy('n.nid');
    $query->groupBy('n.type');
    $query->groupBy('n.langcode');
    if ($min_revisions > 0) {
      $query->having('COUNT(r.vid) >= :min', [':min' => $min_revisions]);
    }
    if ($nid) {
      $query->condition('n.nid', $nid);
    }
    if ($title) {
      $query->condition('n.title', '%' . $title . '%', 'LIKE');
    }
    if (!empty($content_type_filter)) {
      $query->condition('n.type', $content_type_filter);
    }

    // Add sorting.
    $query->orderBy('revision_count', $sort === 'asc' ? 'ASC' : 'DESC');

    // Calc pages.
    $total_items = count($query->execute()->fetchAll());
    $total_pages = ceil($total_items / $items_per_page);

    // Add pager.
    $query = $query->extend(PagerSelectExtender::class)->limit($items_per_page);

    $result = $query->execute();
    $rows = [];

    foreach ($result as $record) {
      $delete_url = Url::fromRoute('node_health.confirm_delete_revisions', [
        'nid' => $record->nid,
      ]);

      $rows[] = [
        'data' => [
          $record->nid,
          Link::fromTextAndUrl($record->title, Url::fromRoute('entity.node.canonical', ['node' => $record->nid]))->toString(),
          $record->type,
          $record->revision_count,
          Link::fromTextAndUrl($this->t('Delete old revisions'), $delete_url)->toString(),
        ],
      ];
    }

    $pagination_items = [];
    // Always show link to first page if not in range.
    if ($current_page > 2) {
      $url = Url::fromRoute('<current>', [], [
        'query' => [
          'min' => $min_revisions,
          'content_type' => $content_type_filter,
          'items_per_page' => $items_per_page,
          'page' => 0,
        ],
      ]);
      $pagination_items[] = Link::fromTextAndUrl('First', $url);
    }

    // Pages around current (current -2 to current +2)
    $start = max(0, $current_page - 2);
    $end = min($total_pages - 1, $current_page + 2);

    for ($i = $start; $i <= $end; $i++) {
      $url = Url::fromRoute('<current>', [], [
        'query' => [
          'min' => $min_revisions,
          'content_type' => $content_type_filter,
          'items_per_page' => $items_per_page,
          'page' => $i,
        ],
      ]);
      $text = (string) ($i + 1);
      if ($i == $current_page) {
        $pagination_items[] = [
          '#markup' => '<strong>' . $text . '</strong>',
        ];
      }
      else {
        $pagination_items[] = Link::fromTextAndUrl($text, $url);
      }
    }

    // Always show link to last page if not in range.
    if ($current_page < $total_pages - 3) {
      $url = Url::fromRoute('<current>', [], [
        'query' => [
          'page' => $total_pages - 1,
        ],
      ]);
      $pagination_items[] = Link::fromTextAndUrl('Last', $url);
    }

    return [
      '#type' => 'fieldset',
      '#title' => $this->t('Node Revisions'),
      '#attributes' => ['id' => "node-health-revision"],
      'cleanup_link' => [
        '#type' => 'link',
        '#title' => $this->t('Batch Cleanup Revisions'),
        '#url' => Url::fromRoute('node_health.revision_cleanup'),
        '#attributes' => [
          'class' => ['button', 'button--primary'],
        ],
      ],
      'form' => [
        '#type' => 'form',
        '#method' => 'get',
        'filters' => $filters,
      ],
      '#attached' => [
        'library' => [
          'node_health/node_health.admin',
        ],
      ],
      
      'table_container' => [
        '#type' => 'container',
        'table_counter' => [
          '#type' => "container",
          '#attributes' => [
            'class' => ['table-counter'],
          ],
          '#markup' => 'Showing page ' . ($current_page + 1) . ' of ' . $total_pages . ' of ' . $total_items . ' results',
        ],
        'table' => [
          '#type' => 'table',
          '#header' => $header,
          '#rows' => $rows,
          '#empty' => $this->t('No nodes found.'),
        ],
      ],
      'pagination' => [
        '#theme' => 'item_list',
        '#items' => $pagination_items,
        '#attributes' => [
          'class' => ['pagination', 'pagination-center'],
        ],
      ],
    ];
  }

  /**
   * Confirmation form for deleting old revisions.
   *
   * @param int $nid
   *   Node ID.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   Render array or redirect response.
   */
  public function confirmDelete($nid) {
    $request = $this->requestStack->getCurrentRequest();
    $node = $this->entityTypeManager->getStorage('node')->load($nid);
    if (!$node) {
      $this->messenger->addError($this->t('Node not found.'));
      return $this->redirect('node_health.report');
    }

    $min = (int) $request->query->get('min', 1);
    $op = $request->query->get('op', NULL);

    if ($nid && $min && $op == 'Confirm and delete old revisions') {
      $url = Url::fromRoute('node_health.delete_node_revisions', [
        'nid' => $nid,
        'min' => $min,
      ])->toString();

      return new RedirectResponse($url);
    }

    $storage = $this->entityTypeManager->getStorage('node');
    $revision_ids = $storage->revisionIds($node);
    // Newest first.
    $revision_ids = array_reverse($revision_ids);

    $rows = [];
    foreach ($revision_ids as $key => $vid) {
      $revision = $storage->loadRevision($vid);
      if ($revision) {
        $rows[] = [
          '' => $key + 1,
          'vid' => $vid,
          'date' => \Drupal::service('date.formatter')->format($revision->getRevisionCreationTime(), 'short'),
          'log' => $revision->getRevisionLogMessage(),
          'default' => $revision->isDefaultRevision() ? 'TRUE' : 'FALSE',
        ];
      }
    }

    $form = [];
    $form['#type'] = 'form';
    $form['#method'] = 'get';

    $form['notice'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning"><strong>Note:</strong> Only the latest revisions will be preserved. Older revisions may be removed during the process.</div>',
    ];
    $form['min'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of latest revisions to keep'),
      '#default_value' => 1,
      '#value' => $min,
      '#attributes' => ['name' => 'min'],
      '#min' => 1,
      '#max' => count($revision_ids),
      '#required' => TRUE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Confirm and delete old revisions'),
      '#button_type' => 'primary',
    ];

    $form['nid'] = ['#type' => 'hidden', '#attributes' => ['name' => 'nid'], '#value' => $nid];

    return [
      'title' => ['#markup' => $this->t('Revisions of: @title', ['@title' => $node->label()])],
      'form' => $form,
      'space' => [
        '#markup' => Markup::create('<br>'),
      ],
      'revisions_table' => [
        '#type' => 'table',
        '#header' => ['', $this->t('Revision ID'), $this->t('Date'), $this->t('Message'), $this->t('Default')],
        '#rows' => $rows,
        '#empty' => $this->t('No revisions found.'),
      ],
    ];
  }

  /**
   * Delete old revisions and keep how many the user chose.
   *
   * @param int $nid
   *   Node ID.
   * @param int $min
   *   Number of latest revisions to keep.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response.
   */
  public function deleteOldRevisions(int $nid, int $min) {
    $request = $this->requestStack->getCurrentRequest();
    $node = $this->entityTypeManager->getStorage('node')->load($nid);
    $default_revision_id = $node->getRevisionId();

    if (!$node || $min < 1) {
      $this->messenger->addError($this->t('Invalid node or filter value.'));
      return $this->redirect('node_health.node_revision_report');
    }

    $revision_ids = $this->entityTypeManager
      ->getStorage('node')
      ->revisionIds($node);

    $revision_ids = array_combine($revision_ids, $revision_ids);

    // Remove the default revision from the list to avoid deleting it.
    unset($revision_ids[$default_revision_id]);
    // Sort descending to keep the latest first.
    rsort($revision_ids);

    if (count($revision_ids) > $min) {
      // Keep only the first $min elements, removing from the end.
      $revision_ids = array_slice($revision_ids, $min);
    }

    $batch_builder = (new BatchBuilder())
      ->setTitle($this->t('Deleting old revisions'))
      ->setInitMessage($this->t('Starting deletion...'))
      ->setProgressMessage($this->t('Deleting revision @current of @total'))
      ->setErrorMessage($this->t('Deletion has encountered an error.'))
      ->setFinishCallback([static::class, 'onBatchFinish']);

    foreach ($revision_ids as $vid) {
      $batch_builder->addOperation([static::class, 'deleteSingleRevision'], [$nid, $vid]);
    }

    batch_set($batch_builder->toArray());

    return batch_process(Url::fromRoute('node_health.report', ['query' => ['min' => $min]]));
  }

  /**
   * Method called by the batch process to remove the revision.
   *
   * @param int $nid
   *   Node ID.
   * @param int $vid
   *   Revision ID.
   * @param array $context
   *   Batch context.
   */
  public static function deleteSingleRevision($nid, $vid, &$context) {
    $entityTypeManager = \Drupal::service('entity_type.manager');
    $node_storage = $entityTypeManager->getStorage('node');
    $node_revision = $node_storage->loadRevision($vid);

    $node = $node_storage->load($nid);

    if ($node_revision && $node_revision->id() == $nid && !$node_revision->isDefaultRevision() && $node->getRevisionId() != $node_revision->getRevisionId()) {
      $node_storage->deleteRevision($vid);
    }

    $context['message'] = t('Deleted revision @vid of node @nid', ['@vid' => $vid, '@nid' => $nid]);
  }

  /**
   * Added message when the batch finishes.
   */
  public static function onBatchFinish($success, $results, $operations) {
    if ($success) {
      \Drupal::messenger()->addStatus(t('Revisions deleted successfully.'));
    }
    else {
      \Drupal::messenger()->addError(t('There was an error deleting revisions.'));
    }
  }

  /**
   * View Paragraph Fields.
   *
   * @param int $paragraph_id
   *   Paragraph ID.
   * @param int $revision_id
   *   Revision ID.
   *
   * @return array
   *   Render array for the paragraph fields table.
   */
  public function viewParagraphFields($paragraph_id, $revision_id) {
    $paragraph = $this->entityTypeManager
      ->getStorage('paragraph')
      ->loadRevision($revision_id);

    if (!$paragraph) {
      throw new NotFoundHttpException("Paragraph not found.");
    }

    $fields = $paragraph->getFields();
    $rows = [];

    foreach ($fields as $field_name => $field_item_list) {
      if ($field_item_list->access('view')) {
        $field_value = [];

        foreach ($field_item_list as $item) {
          $field_definition = $paragraph->getFieldDefinition($field_name);

          if ($field_definition) {
            // Get the target entity type.
            $settings = $field_definition->getSettings();
            $target_type = $settings['target_type'] ?? NULL;

            if ($target_type === 'paragraph') {
              $paragraph_entity = $item->entity;
              $paragraph_id = $paragraph_entity->id();
              $revision_id = $paragraph_entity->getRevisionId();

              $link = Url::fromRoute('node_health.paragraph_revision_fields', [
                'paragraph_id' => $paragraph_id,
                'revision_id' => $revision_id,
              ])->toString();
              $field_value[] = '<a href="' . $link . '">' . $paragraph_id . ' (rev:' . $revision_id . ')</a>';
            }
          }
          else {
            $field_value[] = $item->value;
          }
        }

        $rows[] = [
          'data' => [
            'field_name' => $field_name,
            'field_value' => implode(',', $field_value),
          ],
        ];
      }
    }
    $header = [
      $this->t('Field name'),
      $this->t('Field value'),
    ];

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No accessible fields found.'),
    ];
  }

}
