<?php

namespace Drupal\node_health\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form for cleaning up old node revisions.
 */
class NodeHealthCleanUpForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_health_cleanup_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Node Revisions Cleanup'),
    ];
    $form['fieldset']['description'] = [
      '#markup' => $this->t('<p><strong>Important:</strong> This operation will process all content on the site and delete older revisions for every node. Only the latest <em>number of revisions</em> you specify below will be kept. This cannot be undone.</p>'),
    ];

    $form['fieldset']['revisions_to_keep'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of latest revisions to keep'),
      '#default_value' => 3,
      '#min' => 1,
      '#required' => TRUE,
    ];

    $form['fieldset']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Start cleanup'),
      '#attributes' => [
        'class' => ['button'],
        'style' => 'margin:0',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $revisions_to_keep = (int) $form_state->getValue('revisions_to_keep');

    $nids = \Drupal::entityQuery('node')
      ->accessCheck(FALSE)
      ->execute();

    $operations = [];

    foreach ($nids as $nid) {
      $operations[] = [
        [static::class, 'processNodeRevisions'],
        [$nid, $revisions_to_keep],
      ];
    }

    $batch = [
      'title' => $this->t('Cleaning up old node revisions...'),
      'operations' => $operations,
      'finished' => [static::class, 'finishedBatch'],
    ];

    batch_set($batch);
  }

  /**
   * Processes node revisions.
   */
  public static function processNodeRevisions($nid, $revisions_to_keep, array &$context) {
    $storage = \Drupal::entityTypeManager()->getStorage('node');
    // Get all revisions of the current node, in all languages.
    $revisions = $storage->revisionIds($storage->load($nid));
    $node = $storage->load($nid);
    $current_revision_id = $node->getRevisionId();
    // Creating an array with the keys equal to the value.
    $revisions = array_combine($revisions, $revisions);
    // Remove the current revision from the list.
    unset($revisions[$current_revision_id]);

    if (!$revisions || count($revisions) <= $revisions_to_keep) {
      return;
    }

    $context['message'] = t('Processing node ID @nid...', ['@nid' => $nid]);

    // Newest first.
    rsort($revisions);
    $revisions_to_delete = array_slice($revisions, $revisions_to_keep);

    if (count($revisions_to_delete) > $revisions_to_keep) {
      // Keep only the first $revisions_to_keep elements, removing from the end.
      $revisions_to_delete = array_slice($revisions_to_delete, $revisions_to_keep);
    }

    // dd($revisions_to_delete, $nid);.
    foreach ($revisions_to_delete as $revision_id) {
      $node_storage = \Drupal::entityTypeManager()->getStorage('node');
      $context['message'] = t('Deleting revision @rid of node @nid...', [
        '@rid' => $revision_id,
        '@nid' => $nid,
      ]);

      try {
        $node_storage->deleteRevision($revision_id);
        \Drupal::logger('node_health')->notice("Deleted revision @rid of node @nid", [
          '@rid' => $revision_id,
          '@nid' => $nid,
        ]);
      }
      catch (\Exception $e) {
        \Drupal::logger('node_health')->error("Failed to delete revision @rid: @msg", [
          '@rid' => $revision_id,
          '@msg' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Batch finished callback.
   */
  public static function finishedBatch($success, $results, $operations) {
    if ($success) {
      \Drupal::messenger()->addStatus(t('Old revisions cleanup completed.'));
    }
    else {
      \Drupal::messenger()->addError(t('There was an error during the cleanup process.'));
    }
  }

}
