<?php

namespace Drupal\node_health\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\node\Entity\Node;
use Drupal\node\NodeInterface;
use Drupal\node_health\Helper\NodeHealthFieldValuePrinter;

/**
 * Form for exploring node revisions and field values.
 */
class NodeHealthExplorerForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_health_explorer_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $selected_nid = $form_state->getValue('node_select');

    $form['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Node Revision Explorer'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];
    $form['fieldset']['node_select'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Choose your node'),
      '#title' => $this->t('Choose your node'),
      '#target_type' => 'node',
      '#default_value' => NULL,
      '#ajax' => [
        'callback' => '::updateRevisions',
        'wrapper' => 'revision-wrapper',
        'event' => 'autocompleteclose',
      ],
    ];

    $form['fieldset']['revision_section'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'revision-wrapper'],
    ];

    if ($selected_nid && $node = Node::load($selected_nid)) {
      $revisions = \Drupal::entityTypeManager()
        ->getStorage('node')
        ->revisionIds($node);
      // Latest first.
      $revisions = array_reverse($revisions);

      $options = [];
      $options[] = '';
      foreach ($revisions as $vid) {
        $revision = \Drupal::entityTypeManager()->getStorage('node')->loadRevision($vid);
        $options[$vid] = $this->t('Revision @vid (Updated: @date)', [
          '@vid' => $vid,
          '@date' => \Drupal::service('date.formatter')->format($revision->getRevisionCreationTime(), 'short'),
        ]);
      }

      $form['revision_section']['revision_select'] = [
        '#type' => 'select',
        '#title' => $this->t('Select a revision'),
        '#options' => $options,
        '#ajax' => [
          'callback' => '::showRevisionFields',
          'event' => 'change',
          'wrapper' => 'field-links-wrapper',
        ],
      ];
    }

    $form['revision_section']['fields_output'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'field-links-wrapper'],
    ];

    return $form;
  }

  /**
   * Updates the revision section.
   */
  public function updateRevisions(array &$form, FormStateInterface $form_state) {
    return $form['revision_section'];
  }

  /**
   * Shows the fields of the selected revision.
   */
  public function showRevisionFields(array &$form, FormStateInterface $form_state) {

    $triggering_element = $form_state->getTriggeringElement();
    $revision_id = $triggering_element['#value'];

    $form['revision_section']['fields_output'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'field-links-wrapper'],
    ];

    if ($revision_id) {
      $rev_node = \Drupal::entityTypeManager()->getStorage('node')->loadRevision($revision_id);
      if ($rev_node instanceof NodeInterface) {
        foreach ($rev_node->getFields() as $field_name => $field) {
          if ($field->access('view')) {
            $field_definition = $field->getFieldDefinition();
            $field_type = $field_definition->getType();
            $settings = $field_definition->getSettings();

            $field_value = NodeHealthFieldValuePrinter::printFieldValue($field);

            $rows[] = [
              'data' => [
                'field_name' => $field_name,
                'field_value' => [
                  'data' => [
                    '#markup' => Markup::create($field_value),
                  ],
                ],
              ],
            ];  

          }
        }
      }
    }

    $header = [
      $this->t('Field name'),
      $this->t('Field value'),
    ];

    $form['revision_section']['fields_output']['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No accessible fields found.'),
    ];

    return $form['revision_section']['fields_output'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // No submit needed for this form.
  }

}
