<?php

namespace Drupal\node_health\Form;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to optimize node tables.
 */
class NodeHealthOptimizeTablesForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_health_optimize_tables_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['description'] = [
      '#markup' => '<p>This will run the command <b>OPTIMIZE TABLE</b> on selected <b>node tables</b> in the database.</p>',
    ];

    // Fetch list of tables matching "node%".
    $connection_info = $this->database->getConnectionOptions();
    $database_name = $connection_info['database'];

    $query = $this->database->select('information_schema.TABLES', 't')
      ->fields('t', ['TABLE_NAME'])
      ->condition('t.TABLE_SCHEMA', $database_name)
      ->condition('t.ENGINE', 'InnoDB')
      ->condition('t.TABLE_NAME', 'node%', 'LIKE');

    $tables = $query->execute()->fetchCol();

    if (empty($tables)) {
      $form['no_tables'] = ['#markup' => $this->t('No node tables found.')];
      return $form;
    }

    $table_options = array_combine($tables, $tables);

    $form['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Optimize Node Tables'),
    ];

    $form['fieldset']['filter_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['node-health-optimize-filter-wrapper']],
    ];

    $form['fieldset']['filter_wrapper']['filter'] = [
      '#type' => 'textfield',
      '#attributes' => [
        'placeholder' => $this->t('Filter tables...'),
        'class' => ['node-health-optimize-filter'],
      ],
      '#title_display' => 'invisible',
    ];

    $form['fieldset']['actions_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['node-health-optimize-actions']],
      '#markup' => Markup::create('
        <div class="node-health-optimize-actions">
          <button type="button" class="node-health-optimize-select-all button button--small">Select All</button>
          <button type="button" class="node-health-optimize-toggle button button--small">Toggle Selection</button>
          <button type="button" class="node-health-optimize-clear-all button button--small">Clear All</button>
        </div>
      <br>'),
    ];

    $form['fieldset']['tables'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select tables to optimize'),
      '#options' => $table_options,
      '#default_value' => [],
      '#prefix' => '<div id="node-health-table-wrapper">',
      '#suffix' => '</div>',
    ];

    $form['fieldset']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Start Optimization'),
      '#button_type' => 'primary',
    ];

    $form['#attached']['library'][] = 'node_health/node_health.optimize_form';

    return $form;
  }

  /**
   * AJAX callback to select all tables.
   */
  public function ajaxSelectAll(array &$form, FormStateInterface $form_state) {
    $tables = array_keys($form['tables']['#options']);
    $form_state->setValue('tables', array_combine($tables, $tables));
    $form['tables']['#value'] = array_combine($tables, $tables);
    return $form['tables'];
  }

  /**
   * AJAX callback to clear all selections.
   */
  public function ajaxClearAll(array &$form, FormStateInterface $form_state) {
    $form_state->setValue('tables', []);
    $form['tables']['#value'] = [];
    return $form['tables'];
  }

  /**
   * AJAX callback to toggle selections.
   */
  public function ajaxToggleSelected(array &$form, FormStateInterface $form_state) {
    $current_values = $form_state->getValue('tables');
    $toggled = [];
    foreach ($form['tables']['#options'] as $key => $label) {
      $toggled[$key] = empty($current_values[$key]) ? $key : 0;
    }
    $form_state->setValue('tables', $toggled);
    $form['tables']['#value'] = $toggled;
    return $form['tables'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $selected_tables = array_filter($form_state->getValue('tables'));

    if (empty($selected_tables)) {
      $this->messenger()->addError($this->t('Please select at least one table.'));
      return;
    }

    $batch = (new BatchBuilder())
      ->setTitle($this->t('Optimizing Database Tables...'))
      ->setInitMessage($this->t('Starting optimization...'))
      ->setProgressMessage($this->t('Optimizing...'))
      ->setErrorMessage($this->t('An error occurred during optimization.'))
      ->setFinishCallback([static::class, 'batchFinished']);

    foreach ($selected_tables as $table) {
      $batch->addOperation([static::class, 'optimizeInnoDBTable'], [$table]);
    }

    batch_set($batch->toArray());
  }

  /**
   * Optimize a single InnoDB table.
   */
  public static function optimizeInnoDBTable($table_name, &$context) {
    try {
      \Drupal::database()->query("ALTER TABLE `$table_name` ENGINE = InnoDB");
      $context['results']['optimized'][] = $table_name;
      $context['message'] = t('Optimized @table', ['@table' => $table_name]);
    }
    catch (\Exception $e) {
      $context['results']['errors'][] = $table_name;
      \Drupal::logger('node_health')->error('Failed to optimize @table: @error', [
        '@table' => $table_name,
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Batch finished callback.
   */
  public static function batchFinished($success, array $results, array $operations) {
    if ($success) {
      $count = count($results['optimized'] ?? []);
      \Drupal::messenger()->addMessage(t('Optimization complete. @count tables optimized. Tables: @tables', [
        '@count' => $count,
        '@tables' => implode(', ', $results['optimized']),
      ]));
    }
    else {
      \Drupal::messenger()->addError(t('Optimization encountered errors.'));
    }

    if (!empty($results['errors'])) {
      \Drupal::messenger()->addWarning(t('Some tables failed to optimize: @tables', [
        '@tables' => implode(', ', $results['errors']),
      ]));
    }
  }

}
