<?php

namespace Drupal\node_health\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form to drop node_r__ tables.
 */
class NodeHealthOrphanRTablesConfirmForm extends ConfirmFormBase {

  /**
   * Temp store service.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStore;

  public function __construct(PrivateTempStoreFactory $temp_store_factory) {
    $this->tempStore = $temp_store_factory->get('node_health');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tempstore.private')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_health_drop_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to drop the selected tables?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return Url::fromRoute('<current>');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This action cannot be undone. Make sure you have a backup.');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $tables = $this->tempStore->get('tables_to_drop') ?? [];

    if (empty($tables)) {
      $this->messenger()->addWarning($this->t('No tables selected.'));
      $form_state->setRedirect('node_health.orphan.r.tables');
      return [];
    }

    $form = parent::buildForm($form, $form_state);

    $form['tables'] = [
      '#theme' => 'item_list',
      '#items' => $tables,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $tables = $this->tempStore->get('tables_to_drop') ?? [];

    foreach ($tables as $table) {
      \Drupal::database()->query("DROP TABLE `$table`");
      $this->messenger()->addStatus($this->t('Dropped table @table.', ['@table' => $table]));
    }

    $this->tempStore->delete('tables_to_drop');
    $this->messenger()->addStatus($this->t('Selected tables deleted successfully.'));
    $form_state->setRedirect('node_health.orphan.r.tables');
  }

}
