<?php

namespace Drupal\node_health\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node_health\Helper\NodeHealthFormatSize;

/**
 * Form to show and delete orphaned node_r__* tables with confirmation.
 */
class NodeHealthOrphanRTablesForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'node_health_drop_r_tables_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $confirm_step = $form_state->get('confirm_step');

    $queue_items = \Drupal::service('queue')->get('entity_reference_revisions_orphan_purger')->numberOfItems();

    $schema = \Drupal::database()->getConnectionOptions()['database'];
    $all_tables = \Drupal::database()->query("
      SELECT table_name
      FROM information_schema.tables
      WHERE table_schema = :schema
        AND table_name LIKE 'node_r\\_\\_%'
    ", [':schema' => $schema])->fetchCol();

    // Get all node field storage definitions.
    $field_storages = FieldStorageConfig::loadMultiple();
    $used_tables = [];
    $hash_to_field = [];

    foreach ($field_storages as $field_storage) {
      if ($field_storage->getTargetEntityTypeId() !== 'node') {
        continue;
      }

      $field_name = $field_storage->getName();
      $hash = substr(hash('sha256', $field_storage->getUniqueStorageIdentifier() ?? ''), 0, 10);
      $hash_to_field["node_r__{$hash}"] = $field_name;
      $used_tables[$field_name] = 'node_r__' . $hash;
    }

    $unused = array_diff($all_tables, $used_tables);
    $rows = [];
    $total_size = 0;
    $total_rows = 0;

    foreach ($unused as $table) {
      $row_count = \Drupal::database()->query("SELECT COUNT(*) FROM `$table`")->fetchField();
      $size = \Drupal::database()->query("
        SELECT ROUND((data_length + index_length), 2) AS size_mb
        FROM information_schema.tables
        WHERE table_schema = :schema AND table_name = :table
      ", [':schema' => $schema, ':table' => $table])->fetchField();

      // Try to get distinct bundle values from table (if the column exists).
      $bundles = [];
      try {
        $column_check = \Drupal::database()->schema()->fieldExists($table, 'bundle');
        if ($column_check) {
          $bundles = \Drupal::database()->query("SELECT DISTINCT bundle FROM `$table`")->fetchCol();
        }
      }
      catch (\Exception $e) {
        // Silently skip if table has issues or no access.
      }

      $rows[$table] = [
        'table' => $table,
        'bundle' => $bundles,
        'rows' => $row_count,
        'size' => $size,
      ];

      $total_rows += $row_count;
      $total_size += (float) $size;
    }

    $form['#attached']['library'][] = 'node_health/node_health.r_table_bulk_tools';

    if (empty($rows)) {
      $form['message'] = [
        '#markup' => '<p>✅ No orphaned unused node revision tables found.</p>',
      ];
      return $form;
    }

    if ($queue_items > 0) {
      $form['queue_warning'] = [
        '#markup' => '<div class="messages messages--warning">
          <h4>⏳ There are pending items in the queue</h4>
          <p><strong>' . number_format($queue_items) . '</strong> item(s) are still pending in the <code>entity_reference_revisions_orphan_purger</code> queue.</p>
          <p>Before deleting any <code>node_r__*</code> tables, it is recommended to process the queue to avoid database errors:</p>
          <pre><code>drush queue:run entity_reference_revisions_orphan_purger</code></pre>
        </div>',
      ];
    }

    $form['info'] = [
      '#markup' => $this->t('
         <div class="messages messages--warning">
          <h3>⚠️ What are <code>node_r__</code> tables?</h3>
          <p>When you add a field to a content type and enable revisions, Drupal creates a table like <code>node_r__HASH</code> to store past versions of that field for node revisions.</p>

          <h4>🧹 Why are some tables orphaned?</h4>
          <p>These tables may become <strong>orphaned</strong> if a field or content type was deleted, renamed, or altered. In those cases, Drupal leaves behind the revision table even though it\’s no longer used by the system.</p>

          <h4>✅ Is it safe to delete them?</h4>
          <p>Yes, as long as the field is no longer part of any active node type. This form detects unused revision tables, and deleting them can help clean up your database. <strong>Be sure to create a backup before deletion.</strong></p>

          <h4>⏳ But what if there\'s a queue item referencing it?</h4>
          <p>If the table is still referenced in the <code>entity_reference_revisions_orphan_purger</code> queue, you <strong>must process that queue before deleting the table</strong>. Otherwise, you may get database errors like <code>Table doesn\'t exist</code>.</p>
          <p>To process the queue manually, run the following Drush command:</p>
          <pre><code>drush queue:run entity_reference_revisions_orphan_purger</code></pre>
        </div>
      '),
    ];

    $form['filter_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['node-health-r-tables-filter-wrapper']],
    ];

    $form['filter_wrapper']['filter'] = [
      '#type' => 'textfield',
      '#attributes' => [
        'placeholder' => $this->t('Filter tables...'),
        'class' => ['node-health-r-tables-filter'],
      ],
      '#title_display' => 'invisible',
    ];

    // Add JS control buttons.
    $form['controls'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['node-health-orphan-r-tables-actions']],
      '#markup' => Markup::create('
        <div class="node-health-orphan-r-tables-actions">
          <button type="button" class="node-health-orphan-r-tables-select-all button button--small">Select All</button>
          <button type="button" class="node-health-orphan-r-tables-toggle button button--small">Toggle Selection</button>
          <button type="button" class="node-health-orphan-r-tables-clear-all button button--small">Clear All</button>
        </div>
      <br>'),
    ];

    $form['table_counter'] = [
      '#type' => "container",
      '#attributes' => [
        'class' => ['table-counter'],
      ],
      '#markup' => 'Showing ' . count($unused) . ' tables with ' . $total_rows . ' rows.',
    ];

    $form['tables'] = [
      '#type' => 'table',
      '#header' => ['Select', 'Table name', 'Row count', 'Size (MB)'],
    ];

    foreach ($rows as $table => $info) {
      $form['tables'][$table]['select'] = [
        '#type' => 'checkbox',
        '#attributes' => ['class' => ['drop-table-checkbox'], 'table-name' => $table],
      ];
      $form['tables'][$table]['name'] = ['#markup' => $info['table']];
      $form['tables'][$table]['bundle'] = ['#markup' => !empty($info['bundle']) ? implode(', ', $info['bundle']) : '<em>—</em>'];
      $form['tables'][$table]['rows'] = ['#markup' => number_format($info['rows'])];
      $form['tables'][$table]['size'] = ['#markup' => NodeHealthFormatSize::formatSize($info['size'])];
    }

    $form['tables']['_total'] = [
      'select' => ['#markup' => ''],
      'name' => ['#markup' => '<strong>Total</strong>'],
      'rows' => ['#markup' => '—'],
      'rows' => ['#markup' => '—'],
      'size' => ['#markup' => '<strong>' . NodeHealthFormatSize::formatSize(round($total_size, 2)) . '</strong>'],
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $selected = array_filter($form_state->getValue('tables') ?? [], function ($val) {
      return !empty($val['select']);
    });

    if (empty($selected)) {
      $this->messenger()->addWarning($this->t('No tables selected.'));
      return;
    }

    $table_names = array_keys($selected);
    \Drupal::service('tempstore.private')->get('node_health')->set('tables_to_drop', $table_names);

    $form_state->setRedirect('node_health.orphan.confirm');
  }

}
