<?php

namespace Drupal\node_health\Helper;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Url;
use Drupal\file\Entity\File;

/**
 * Class NodeHealthFieldValuePrinter.
 */
final class NodeHealthFieldValuePrinter {

  /**
   * Auto-detect field type and print its value.
   */
  public static function printFieldValue(FieldItemListInterface $field) {
    $field_definition = $field->getFieldDefinition();
    $field_type = $field_definition->getType();

    if ($field_type == 'path') {
      // dd($field->value);.
    }
    switch ($field_type) {
      case 'path':
      case 'uuid':
      case 'created':
      case 'changed':
      case 'language':
      case 'string':
      case 'string_long':
        return self::printPlainText($field);

      case 'metatag_computed':
        return self::printArray($field);

      case 'metatag':
        return self::printPlainText($field);

      case 'map':
        return self::printArray($field);

      case 'text':
      case 'text_long':
      case 'text_with_summary':
        return self::printFormattedText($field);

      case 'integer':
      case 'decimal':
      case 'float':
        return self::printNumber($field);

      case 'entity_reference':
        $target_type = $field_definition->getSetting('target_type');
        if ($target_type === 'paragraph') {
          return self::printParagraphs($field);
        }
        elseif ($target_type === 'media') {
          return self::printMedia($field);
        }
        elseif ($target_type === 'file') {
          return self::printFile($field);
        }
        else {
          return self::printReference($field);
        }

      case 'entity_reference_revisions':
        $target_type = $field_definition->getSetting('target_type');
        if ($target_type === 'paragraph') {
          return self::printParagraphs($field);
        }
        else {
          return self::printReference($field);
        }

      case 'list_string':
      case 'list_integer':
      case 'list_float':
        return self::printSelectionList($field);

      case 'datetime':
      case 'timestamp':
        return self::printDateTime($field);

      case 'boolean':
        return self::printBoolean($field);

      case 'email':
        return self::printEmail($field);

      case 'link':
        return self::printLink($field);

      case 'image':
        return self::printImage($field);

      default:
        return 'Unsupported field type (' . $field_type . ')';
    }
  }

  /**
   * Prints an array in a readable format.
   *
   * @param array $array
   */
  public static function printArray($field) {
    $value = $field->getValue();
    if ($value) {
      return '<pre>' . print_r($value, TRUE) . '</pre>';
    }
  }

  /**
   * Render an image field as <img>.
   */
  public static function printImage(FieldItemListInterface $field) {
    $output = [];

    $file_url_generator = \Drupal::service('file_url_generator');

    foreach ($field as $item) {
      /** @var \Drupal\file\FileInterface $file */
      $file = $item->entity;
      if ($file instanceof File) {
        $url = $file_url_generator->generateAbsoluteString($file->getFileUri());
        $output[] = '<img src="' . $url . '" alt="" style="max-width: 150px;">';
      }
    }

    return implode('<br>', $output);
  }

  /**
   * Plain text field.
   */
  public static function printPlainText(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      return $item->value;
    });
  }

  /**
   * Formatted text field.
   */
  public static function printFormattedText(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      return $item->value;
    });
  }

  /**
   * Number field.
   */
  public static function printNumber(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      return $item->value;
    });
  }

  /**
   * Reference field (node, taxonomy, user, etc.).
   */
  public static function printReference(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      $entity = $item->entity;
      return $entity ? $entity->label() : '';
    });
  }

  /**
   * Media field.
   */
  public static function printMedia(FieldItemListInterface $field) {
    $output = [];

    foreach ($field as $item) {
      if ($item->entity && $item->entity->bundle() === 'image') {
        $media = $item->entity;
        $image_field = $media->get('field_media_image');

        if (!$image_field->isEmpty()) {
          $file = $image_field->entity;
          if ($file) {
            $url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
            $output[] = '<img src="' . $url . '" alt="' . htmlspecialchars($media->label()) . '" style="max-width: 150px;"/>';
          }
        }
        else {
          $output[] = htmlspecialchars($media->label()) . ' (No image file)';
        }
      }
      elseif ($item->entity) {
        $output[] = htmlspecialchars($item->entity->label()) . ' (Not an image)';
      }
    }

    return implode('<br>', $output);
  }

  /**
   * File upload field.
   */
  public static function printFile(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      /** @var \Drupal\file\Entity\File $file */
      $file = File::load($item->target_id);
      return $file ? $file->createFileUrl() : '';
    });
  }

  /**
   * Paragraphs field.
   */
  public static function printParagraphs(FieldItemListInterface $field) {
    $output = [];

    foreach ($field as $item) {
      if ($item->entity) {
        $paragraph = $item->entity;
        $paragraph_id = $paragraph->id();
        $revision_id = $paragraph->getRevisionId();

        if (!empty($paragraph_id) && !empty($revision_id)) {
          $link = Url::fromRoute('node_health.paragraph_revision_fields', [
            'paragraph_id' => $paragraph_id,
            'revision_id' => $revision_id,
          ])->toString();

          $output[] = '<a href="' . $link . '" target="_blank">' . htmlspecialchars($paragraph->bundle()) . ' pid:' . $paragraph_id . ' (rev:' . $revision_id . ')</a>';
        }
        else {
          // Fallback when ID or Revision is missing.
          $output[] = htmlspecialchars($paragraph->bundle()) . ' (No ID or revision)';
        }
      }
    }

    return implode('<br>', $output);
  }

  /**
   * Selection list field (list_string, list_integer, list_float).
   */
  public static function printSelectionList(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      return $item->value;
    });
  }

  /**
   * Date and time field.
   */
  public static function printDateTime(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      return $item->value;
    });
  }

  /**
   * Boolean field.
   */
  public static function printBoolean(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      return (bool) $item->value ? 'Yes' : 'No';
    });
  }

  /**
   * Email field.
   */
  public static function printEmail(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      return $item->value;
    });
  }

  /**
   * Link field.
   */
  public static function printLink(FieldItemListInterface $field) {
    return self::printMultipleValues($field, function ($item) {
      /** @var \Drupal\link\LinkItemInterface $item */
      $uri = $item->uri;
      $title = $item->title;
      return $title ? "$title ($uri)" : $uri;
    });
  }

  /**
   * Helper function to handle multiple or single values.
   */
  protected static function printMultipleValues(FieldItemListInterface $field, callable $callback) {
    $output = [];
    foreach ($field as $item) {
      $output[] = $callback($item);
    }
    return implode(', ', array_filter($output));
  }

}
