<?php

namespace Drupal\node_health\Commands;

use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Helper\ProgressBar;

/**
 *
 */
class NodeHealthNodeTablesCommands extends DrushCommands {

  /**
   * Runs the node_health optimize node on database hook.
   *
   * @command node-health:optimize-node-table
   * @aliases nrh:op
   * @usage drush node-health:optimize-node-table
   */
  public function optimizeNodeTables(): void {
    $tables = $this->getNodeTables();

    if (empty($tables)) {
      $this->output()->writeln('No node tables found.');
      return;
    }

    $progress = new ProgressBar($this->output(), count($tables));
    $progress->setFormat(" [%bar%] %percent:3s%% | %current%/%max% tables processed");
    $progress->start();

    foreach ($tables as $table) {
      try {
        \Drupal::database()->query("OPTIMIZE TABLE `$table`");
      }
      catch (\Exception $e) {
        $this->output()->writeln("Error optimizing table $table: " . $e->getMessage());
      }
      $progress->advance();
    }

    $progress->finish();
    $this->output()->writeln('All node tables optimized.');
  }

  /**
   * Runs the node_health optimize node on database hook.
   *
   * @command node-health:defragment-node-table
   * @aliases nrh:dpt
   * @usage drush node-health:defragment-node-table
   */
  public function defragmentNodeTables(): void {
    $tables = $this->getNodeTables2();

    if (empty($tables)) {
      $this->output()->writeln('No node tables found.');
      return;
    }

    $progress = new ProgressBar($this->output(), count($tables));
    $progress->setFormat(" [%bar%] %percent:3s%% | %current%/%max% tables processed");
    $progress->start();

    $this->output()->writeln("defragment tables: " . join(',', $tables));
    foreach ($tables as $table) {
      try {
        \Drupal::database()->query("ALTER TABLE `$table` ENGINE = InnoDB");
      }
      catch (\Exception $e) {
        $this->output()->writeln("Error optimizing table $table: " . $e->getMessage());
      }
      $progress->advance();
    }

    $progress->finish();
    $this->output()->writeln('All node tables defragment.');
  }

  /**
   * Runs the node_health optimize Nodes on database hook.
   *
   * @command node-health:show-size-to-optimize
   * @aliases nrh:dpt
   * @usage drush node-health:show-size-to-optimize
   */
  public function showSizeOptimizeTables(): void {
    $result = $this->getNodeTables3();

    $this->output()->writeln("Possible optimization size: $result");
  }

  /**
   * Returns the list of node tables.
   */
  protected function getNodeTables() {
    $database_name = \Drupal::database()->getConnectionOptions()['database'];
    $query = "SELECT TABLE_NAME
      FROM INFORMATION_SCHEMA.TABLES
      WHERE TABLE_SCHEMA = :schema
      AND TABLE_NAME LIKE 'node%'";

    $results = \Drupal::database()->query($query, [':schema' => $database_name])->fetchAll();

    return array_map(fn($row) => current((array) $row), $results);
  }

  /**
   * Returns the list of node tables with free space in MB.
   */
  protected function getNodeTables2() {
    $database_name = \Drupal::database()->getConnectionOptions()['database'];

    $query = "
      SELECT table_name, data_length, index_length, data_free, ROUND(data_free / 1024 / 1024, 2) AS free_MB FROM information_schema.tables WHERE table_schema = :schema AND engine = 'InnoDB' AND table_name LIKE 'node%' AND data_free > 0 ORDER BY free_MB DESC
    ";

    $results = \Drupal::database()->query($query, [':schema' => $database_name])->fetchAll();

    return array_map(fn($row) => current((array) $row), $results);
  }

  /**
   * Returns the total size of free space in MB for all node tables.
   */
  protected function getNodeTables3() {
    $database_name = \Drupal::database()->getConnectionOptions()['database'];

    $query = "
      SELECT 
        ROUND(SUM(data_free) / 1024 / 1024, 2) AS total_data_free_MB
        FROM information_schema.tables
        WHERE table_schema = :schema
          AND data_free > 0 AND table_name LIKE 'node%';
    ";

    $results = \Drupal::database()->query($query, [':schema' => $database_name])->fetchAll();

    return reset($results)->total_data_free_MB;
  }

}
