<?php

namespace Drupal\node_health\Commands;

use Drush\Commands\DrushCommands;

/**
 * Drush command to show node health table sizes.
 */
class NodeHealthTableSizesCommands extends DrushCommands {

  /**
   * Show sizes of node-related tables.
   *
   * @command node--revision-health:show-sizes
   * @aliases nrh:ss
   */
  public function showSizes(array $options = ['tables' => '']) {
    $tables = $options['tables'] ?? '';

    if ($tables) {
      $tables = explode(',', $tables);
    }

    $this->output()->writeln("<info>Node Revision Health Table Sizes:</info>");
    $this->printSizes('node_health_sizes', 'node_health_table_names', $tables);
  }

  /**
   * Prints the sizes of tables from the specified size and name tables.
   */
  private function printSizes(string $size_table, string $name_table, $tables = ''): void {
    $query = \Drupal::database()->select($size_table, 's');

    $query->fields('s', ['size', 'timestamp'])
      ->fields('t', ['table_name'])
      ->innerJoin($name_table, 't', 's.table_id = t.id');
    $query->orderBy('s.size', 'DESC');

    if ($tables) {
      $query->condition('t.table_name', $tables, 'IN');
    }

    $results = $query->execute()->fetchAll();

    if (empty($results)) {
      $this->output()->writeln("  No data found.");
      return;
    }

    $seen = [];
    foreach ($results as $record) {
      if (!isset($seen[$record->table_name])) {
        $size_kb = number_format($record->size / 1024, 2);
        $time = date('Y-m-d H:i:s', $record->timestamp);
        $this->output()->writeln("  <comment>{$record->table_name}</comment> - {$size_kb} KB at {$time}");
        $seen[$record->table_name] = TRUE;
      }
    }
  }

}
