<?php

namespace Drupal\node_health\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Render\Markup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a chart of node content type usage.
 */
class NodeHealthTypeUsageChartController extends ControllerBase {

  protected Connection $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database')
    );
  }

  /**
   * Displays a pie chart of node content type usage.
   */
  public function chart() {
    $query = \Drupal::database()->select('node_field_data', 'n')
      ->fields('n', ['type'])
      ->condition('n.status', 1)
      ->groupBy('n.type');
    $query->addExpression('COUNT(n.nid)', 'count');

    $results = $query->execute()->fetchAllKeyed(0, 1);

    $labels = [];
    $data = [];

    foreach ($results as $type => $count) {
      $labels[] = $type;
      $data[] = (int) $count;
    }

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['node-content-type-usage-chart']],
      'description' => [
        '#markup' => '<p>This chart shows how many nodes exist in each content type.</p>',
      ],
      'chart' => [
        '#markup' => Markup::create('<canvas id="nodeHealthTypeUsageChart" width="400" height="400"></canvas>'),
      ],
      '#attached' => [
        'library' => [
          'node_health/node_health.usage',
        ],
        'drupalSettings' => [
          'node_health' => [
            'labels' => $labels,
            'data' => $data,
          ],
        ],
      ],
    ];
  }

}
