<?php

namespace Drupal\node_health\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\node_health\Helper\NodeHealthFormatSize;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Render\RendererInterface;

/**
 * Controller for Node Health reports and statistics.
 */
class NodeHealthController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a NodeHealthController object.
   */
  public function __construct(
    Connection $database,
    EntityTypeManagerInterface $entityTypeManager,
    RendererInterface $renderer
  ) {
    $this->database = $database;
    $this->entityTypeManager = $entityTypeManager;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('entity_type.manager'),
      $container->get('renderer')
    );
  }

  /**
   * Displays the Node Health report with content types, field sizes, and stats.
   *
   * @return array
   *   Render array for the Node Health report.
   */
  public function list() {
    $header = [
      'type' => [
        'data' => [
          'container' => [
            "#type" => 'container',
            "#attributes" => [
              "id" => ['node-health-header-title'],
            ],
            'data' => [
              '#markup' => $this->t('Node Content Types<button type="button" id="node-health-export" class="node-health-export button button--primary">Export</button>'),
            ],
          ],
        ],
      ],
    ];

    // Count of current nodes.
    $total_node_items = $this->database->select('node', 'n')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Count of node revisions.
    $total_nodes_revisions = $this->database->select('node_revision', 'nr')
      ->countQuery()
      ->execute()
      ->fetchField();

    $query_log_size = $this->database->query(
      "SELECT ROUND(SUM(data_length + index_length), 2) AS size_mb 
      FROM information_schema.tables 
      WHERE table_schema = :schema 
      AND (table_name = 'node_health_sizes' OR table_name = 'node_health_table_names')",
      [':schema' => $this->database->getConnectionOptions()['database']]
    );
    $total_logs_size = $query_log_size->fetchField();

    $rows = [];
    $sub_tables = [];
    $tables_db_size = 0;
    $full_db_size = $this->getDatabaseSize();

    $types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $connection = $this->database;
    $driver = $connection->driver();

    foreach ($types as $type) {
      $type_id = $type->id();

      $field_configs = $this->entityTypeManager
        ->getStorage('field_config')
        ->loadByProperties([
          'entity_type' => 'node',
          'bundle' => $type_id,
        ]);

      $field_rows = [];
      $sum_columns_fields_total_size = 0;
      $sum_table_rows_count = 0;
      $sum_table_columns_count_revision = 0;
      $sum_table_columns_size_revision = 0;
      $sum_table_columns_total_size = 0;

      foreach ($field_configs as $field_config) {
        $sum_row_fields_total_size = 0;
        $field_table = "node__" . $field_config->getName();
        $field_size = $this->getTableSize($field_table, $driver);
        $table_rows_count = $this->getBundleRowCount($field_table, $type_id, 'bundle');
        $sum_table_rows_count += $table_rows_count;
        $sum_row_fields_total_size += $field_size;
        $sum_columns_fields_total_size += $field_size;

        $field_table_revision = "node_revision__" . $field_config->getName();
        $field_size_revision = $this->getTableSize($field_table_revision, $driver);
        $table_rows_count_revision = $this->getBundleRowCount($field_table_revision, $type_id, 'bundle');
        $sum_table_columns_count_revision += $table_rows_count_revision;
        $sum_row_fields_total_size += $field_size_revision;
        $sum_table_columns_size_revision += $field_size_revision;
        $sum_table_columns_total_size += $sum_row_fields_total_size;

        $field_rows[] = [
          Markup::create(
            $field_config->getName() .
            $this->getInfoIcon('Data related to table: ' . $field_table . ' and ' . $field_table_revision)
          ),
          number_format($table_rows_count, 0, '.', ','),
          NodeHealthFormatSize::formatSize($field_size),
          number_format($table_rows_count_revision, 0, '.', ','),
          NodeHealthFormatSize::formatSize($field_size_revision),
          NodeHealthFormatSize::formatSize($sum_row_fields_total_size),
        ];
      }

      $field_rows[] = [
        [
          'data' => Markup::create('<strong>Total</strong>'),
          'colspan' => 1,
        ],
        [
          'data' => Markup::create('<strong>' . number_format($sum_table_rows_count, 0, '.', ',') . '</strong>'),
        ],
        [
          'data' => Markup::create('<strong>' . NodeHealthFormatSize::formatSize($sum_columns_fields_total_size) . '</strong>'),
        ],
        [
          'data' => Markup::create('<strong>' . number_format($sum_table_columns_count_revision, 0, '.', ',') . '</strong>'),
        ],
        [
          'data' => Markup::create('<strong>' . NodeHealthFormatSize::formatSize($sum_table_columns_size_revision) . '</strong>'),
        ],
        [
          'data' => Markup::create('<strong>' . NodeHealthFormatSize::formatSize($sum_table_columns_total_size) . '</strong>'),
        ],
      ];
      $tables_db_size += $sum_table_columns_total_size;

      // Save the sub table separately.
      $sub_tables[$type->label() . ' (' . $type_id . ')'] = [
        '#type' => 'table',
        '#header' => [
          Markup::create($type_id . ' Field'),
          Markup::create('Rows'),
          Markup::create('Size'),
          Markup::create('Revision Rows'),
          Markup::create('Revision Size'),
          Markup::create('Total'),
        ],
        '#rows' => $field_rows,
        '#attributes' => [
          'class' => [
            'node-health-size-tables',
          ],
          'tab-id' => substr($type_id, 0, 30),
        ],
      ];

      $rows[] = [
        'type' => $type->label() . ' (' . $type_id . ')',
      ];
    }

    $build = [];

    $build['container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['node-health-three-columns-wrapper'],
      ],
    ];

    $build['container']['left'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['column', 'left']],
      'content' => [
        '#markup' => Markup::create($this->getGraphIcon($full_db_size, $tables_db_size)),
      ],
    ];

    $build['container']['middle'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['column', 'middle']],
      'content' => [
        '#markup' => Markup::create('<div><p>Database:<strong> ' . NodeHealthFormatSize::formatSize($full_db_size) . '</strong></p><p>Node and Revision:<strong> ' . NodeHealthFormatSize::formatSize($tables_db_size) . '</strong></p><p>Logs:<strong> ' . NodeHealthFormatSize::formatSize($total_logs_size) . '</strong></p></div>'),
      ],
    ];

    $build['container']['right'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['column', 'right']],
      'content' => [
        '#markup' => Markup::create('<div><p>Total Nodes: <strong>' . number_format($total_node_items) . '</strong></p><p>Total Revisions: <strong>' . number_format($total_nodes_revisions) . '</strong></p></div>'),
      ],
    ];

    $build['main_table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => [],
      '#attributes' => ['id' => 'node-health-table'],
    ];

    foreach ($rows as $row) {
      $row_type = $row['type'];
      $row['type'] = Markup::create('<strong class="node-health-type-name">' . $row['type'] . '</strong>');
      $build['main_table']['#rows'][] = $row;

      // Insert the field sub-table after the node type.
      if (isset($sub_tables[$row_type])) {
        $build['main_table']['#rows'][] = [
          '#markup' => [
            'data' => [
              [
                '#markup' => $this->renderer->renderPlain($sub_tables[$row_type]),
              ],
            ],
            'colspan' => 4,
          ],
        ];
      }
    }

    $build['#attached']['library'][] = 'node_health/node_health.report';

    return $build;
  }

  /**
   * Returns the size of a database table.
   *
   * @param string $table
   *   Table name.
   * @param string $driver
   *   Database driver.
   *
   * @return int
   *   Table size in bytes.
   */
  protected function getTableSize(string $table, string $driver): int {
    $connection = $this->database;
    try {
      switch ($driver) {
        case 'mysql':
          $result = $connection->query("SELECT ROUND((data_length + index_length), 2) as total_size
            FROM information_schema.tables
            WHERE table_schema = :schema
              AND table_name = :table", [
                ':schema' => $connection->getConnectionOptions()['database'],
                ':table' => $table,
              ])->fetchField();
          return (int) $result;

        case 'pgsql':
          $result = $connection->query("SELECT pg_total_relation_size(:table)", [':table' => $table])->fetchField();
          return (int) round($result);

        case 'sqlite':
          // SQLite: very basic estimation.
          $exists = $connection->query("SELECT name FROM sqlite_master WHERE type='table' AND name = :table", [':table' => $table])->fetchField();
          return $exists ? 4 : 0;

        default:
          return 0;
      }
    }
    catch (\Exception $e) {
      return 0;
    }
  }

  /**
   * Returns the total database size.
   *
   * @return int|float
   *   Database size in bytes.
   */
  protected function getDatabaseSize() {
    try {
      $database = $this->database;
      $query = $database->query(
        "SELECT 
          ROUND(SUM(data_length + index_length), 2) AS size_mb 
        FROM information_schema.tables 
        WHERE table_schema = :schema",
        [':schema' => $database->getConnectionOptions()['database']]
      );
      $size = $query->fetchField();
      return $size;
    }
    catch (\Throwable $th) {
      return 0;
    }
  }

  /**
   * Get the number of rows for a given bundle in a table.
   *
   * @param string $table
   *   Table name.
   * @param string $bundle
   *   Bundle name.
   * @param string $field_type
   *   Field type column.
   *
   * @return int
   *   Number of rows.
   */
  protected function getBundleRowCount($table, $bundle, $field_type) {
    $database = $this->database;
    try {
      return (int) $database->select($table, 'p')
        ->condition('p.' . $field_type, $bundle)
        ->countQuery()
        ->execute()
        ->fetchField();

    }
    catch (\Exception $e) {
      return 0;
    }
  }

  /**
   * Returns an info icon with tooltip.
   *
   * @param string $tooltip_text
   *   Tooltip text.
   *
   * @return string
   *   HTML markup for the info icon.
   */
  private function getInfoIcon(string $tooltip_text = '') {
    return '<span class="node-health-info-icon" title="' . htmlspecialchars($tooltip_text) . '" style="cursor: help;">
        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
          <path d="M8 1a7 7 0 1 0 0 14A7 7 0 0 0 8 1zM8 0a8 8 0 1 1 0 16A8 8 0 0 1 8 0z"/>
          <path d="m8.93 6.588-2.29.287-.082.38.45.083c.294.07.352.176.288.469l-.738 3.468c-.194.897.105 1.319.808 1.319.545 0 1.013-.252 1.25-.598l.088-.416c-.287.176-.682.287-.887.287-.294 0-.352-.176-.287-.469l.738-3.468c.194-.897-.105-1.319-.808-1.319zm-.93-2.588a1 1 0 1 1-2 0 1 1 0 0 1 2 0z"/>
        </svg>
      </span>';
  }

  /**
   * Returns an SVG graph icon for database/node usage.
   *
   * @param int|float $db_size
   *   Total database size.
   * @param int|float $node_size
   *   Node table size.
   *
   * @return string
   *   SVG markup.
   */
  private function getGraphIcon($db_size, $node_size) {
    $percent = $db_size > 0 ? round(($node_size / $db_size) * 100, 2) : 0;
    return '<svg width="200" height="180" viewBox="0 0 40 45" xmlns="http://www.w3.org/2000/svg">
      <style>
        .circle-bg {
          fill: none;
          stroke: #eee;
          stroke-width: 3.8;
        }
        .circle {
          fill: none;
          stroke: #007BFF;
          stroke-width: 3.8;
          stroke-linecap: round;
          animation: progress 1.5s ease-out forwards;
        }
        .percentage {
          font-size: 0.5em;
          text-anchor: middle;
          fill: #333;
          font-weight: bold;
        }
        @keyframes progress {
          from {
            stroke-dasharray: 0 100;
          }
        }
        .label {
          font-size: 0.2em;
          fill: #333;
        }
        .legend-color {
          width: 1em;
          height: 1em;
        }
      </style>

      <!-- Background (Other DB usage) -->
      <circle class="circle-bg" cx="18" cy="18" r="15.915" />

      <!-- Foreground (Node usage) -->
      <circle class="circle" cx="18" cy="18" r="15.915"
        stroke-dasharray="' . $percent . ' 100" transform="rotate(-90 18 18)" />

      <!-- Percentage label -->
      <text x="18" y="20.5" class="percentage">' . $percent . '%</text>

      <!-- Legend -->
      <!-- Node -->
      <rect x="1" y="38" width="2" height="2" fill="#007BFF" stroke="#007BFF" />
      <text x="4" y="39.6" class="label">Node (' . $percent . '%)</text>

      <!-- Other -->
      <rect x="1" y="41.5" width="2" height="2" fill="#ccc" stroke="#ccc" />
      <text x="4" y="43.1" class="label">Other Tables (' . (100 - $percent) . '%)</text>
    </svg>';
  }

}
