<?php

namespace Drupal\node_health\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for node health export operations.
 */
class NodeHealthExportController extends ControllerBase {

  /**
   * Displays the export form.
   */
  public function export(Request $request) {
    $form['container'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Export'),
    ];

    $form['container']['export_notice'] = [
      '#type' => 'markup',
      '#markup' => t('<div class="messages messages--warning">
        <strong>Notice:</strong> This export will include <strong>all data collected by the cron</strong> related to the <strong>size of tracked database tables</strong>.<br>
        Make sure cron has run at least once to collect data.<br>
        You can verify or configure cron at <a href=":url" target="_blank">Cron settings</a>.
      </div>', [
        ':url' => Url::fromRoute('system.cron_settings')->toString(),
      ]),
    ];

    $form['container']['export'] = [
      '#type' => 'link',
      '#title' => $this->t('Export CSV'),
      '#url' => Url::fromRoute('node_health.export.csv'),
      '#attributes' => [
        'class' => ['button'],
        'style' => 'margin:0',
      ],
    ];

    return $form;
  }

  /**
   * Exports the node health data as a CSV file.
   */
  public function exportCsv() {
    $header = ['Table Name', 'Size (bytes)', 'Timestamp'];

    $query = \Drupal::database()->select('node_health_sizes', 's');
    $query->innerJoin('node_health_table_names', 't', 's.table_id = t.id');
    $query->fields('s', ['size', 'timestamp', 'count']);
    $query->fields('t', ['table_name']);
    $query->orderBy('s.timestamp', 'DESC');
    $results = $query->execute();

    $rows = [];
    foreach ($results as $record) {
      $rows[] = [
        $record->table_name,
        $record->size,
        date('Y-m-d H:i:s', $record->timestamp),
      ];
    }

    $output = fopen('php://temp', 'r+');
    fputcsv($output, $header);
    foreach ($rows as $row) {
      fputcsv($output, $row);
    }
    rewind($output);
    $csv = stream_get_contents($output);
    fclose($output);

    $response = new Response($csv);
    $response->headers->set('Content-Type', 'text/csv');
    $response->headers->set('Content-Disposition', 'attachment; filename="node_health_report.csv"');
    return $response;
  }

}
