<?php

namespace Drupal\node_health\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\NodeTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a controller to list content types with entity count.
 */
class NodeHealthTypeUsageController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a NodeHealthTypeUsageController object.
   */
  public function __construct(Connection $database, RequestStack $request_stack, EntityTypeManagerInterface $entity_type_manager) {
    $this->database = $database;
    $this->requestStack = $request_stack;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('request_stack'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Displays the content types with counts.
   */
  public function content() {
    // Header with sortable columns.
    $header = [
      'type' => [
        'data' => $this->t('Content Type'),
        'specifier' => 'type',
        'sortable' => TRUE,
      ],
      'count' => [
        'data' => $this->t('Count'),
        'specifier' => 'count',
        'sortable' => TRUE,
      ],
      'revision' => [
        'data' => $this->t('Revisions'),
        'specifier' => 'revision',
        'sortable' => FALSE,
      ],
    ];

    // Load content types to get labels.
    $node_type_storage = $this->entityTypeManager->getStorage('node_type');
    $node_types = $node_type_storage->loadMultiple();

    $results = $this->database->query("
      SELECT type, COUNT(*) as count
      FROM node
      GROUP BY type
    ")->fetchAllKeyed();

    $query_type = $this->database->select('node_field_revision', 'nfr');
    $query_type->join('node', 'n', 'n.nid = nfr.nid');

    $query_type->addField('n', 'type', 'bundle');
    $query_type->addExpression('COUNT(nfr.vid)', 'revision_count');
    $query_type->groupBy('n.type');

    $result_type = $query_type->execute()->fetchAllKeyed();

    // Build rows with type, label, count.
    $rows = [];

    foreach ($results as $type => $count) {
      $node_type = $node_types[$type] ?? NULL;
      $label = $node_type instanceof NodeTypeInterface ? $node_type->label() : $type;
      $rows[] = [
        'type' => $label,
        'count' => isset($count) ? (int) $count : 0,
        'revision' => $result_type[$type] ?? 0,
      ];
    }

    // Get current sort from URL.
    $request = $this->requestStack->getCurrentRequest();

    $table_sort = $request ? $request->query->all() : [];

    $sort_by = $table_sort['sort'] ?? 'count';
    $sort_order = (isset($table_sort['order']) && strtolower($table_sort['order']) === 'asc') ? SORT_ASC : SORT_DESC;

    // Sort the rows.
    array_multisort(array_column($rows, $sort_by), $sort_order, $rows);

    $build['container'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content Types'),
      '#attributes' => [
        'class' => ['node-health-usage'],
      ],
    ];
    $build['container']['table_counter'] = [
      '#type' => "container",
      '#attributes' => [
        'class' => ['table-counter'],
      ],
      '#markup' => 'Showing ' . count($rows) . ' content types.',
    ];

    // Build table render array.
    $build['container']['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => [],
      '#empty' => $this->t('No content types found.'),
      '#sticky' => TRUE,
    ];
    $build['#attached']['library'][] = 'node_health/node_health.usage';

    foreach ($rows as $row) {
      $build['container']['table']['#rows'][] = [
        'type' => $row['type'],
        'count' => number_format($row['count'], 0, '.', ','),
        'revision' => number_format($row['revision'], 0, '.', ','),
      ];
    }

    return $build;
  }

}
