<?php

namespace Drupal\node_health\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Queue\QueueFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to process queue items.
 */
class NodeHealthQueueProcessorForm extends FormBase {

  /**
   * The queue factory service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * Constructs the form object.
   */
  public function __construct(QueueFactory $queue_factory) {
    $this->queueFactory = $queue_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('queue')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_health_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $queues = [
      'node_revision_delete' => 'Node Revision Delete',
    ];

    $options = [];
    foreach ($queues as $queue_name => $label) {
      $count = $this->queueFactory->get($queue_name)->numberOfItems();
      $options[$queue_name] = $this->t('@label (@count items)', ['@label' => $label, '@count' => $count]);
    }

    $form['container'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Process Queue'),
    ];

    $form['container']['queue'] = [
      '#type' => 'select',
      '#title' => $this->t('Select queue to process'),
      '#options' => $options,
      '#required' => TRUE,
    ];

    $form['container']['actions']['#type'] = 'actions';
    $form['container']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Run Queue'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $queue_name = $form_state->getValue('queue');
    $queue = $this->queueFactory->get($queue_name);
    $count = $queue->numberOfItems();

    if ($count === 0) {
      $this->messenger()->addWarning($this->t('The selected queue "@queue" has no items to process.', ['@queue' => $queue_name]));
      return;
    }

    $operations = [];
    for ($i = 0; $i < $count; $i++) {
      $operations[] = [
        [static::class, 'processQueueItem'],
        [$queue_name],
      ];
    }

    $batch = [
      'title' => $this->t('Processing queue: @queue', ['@queue' => $queue_name]),
      'operations' => $operations,
      'finished' => [static::class, 'finishBatch'],
      'progress_message' => $this->t('Processed @current of @total items...'),
    ];

    batch_set($batch);
  }

  /**
   * Batch callback: process one item per operation.
   */
  public static function processQueueItem($queue_name, &$context) {
    $queue = \Drupal::service('queue')->get($queue_name);
    if ($item = $queue->claimItem()) {
      try {
        $worker = \Drupal::service('plugin.manager.queue_worker')->createInstance($queue_name);
        $worker->processItem($item->data);
        $queue->deleteItem($item);
      }
      catch (\Exception $e) {
        \Drupal::logger('node_health')->error('Failed processing item: @message', ['@message' => $e->getMessage()]);
      }
    }
  }

  /**
   * Finish batch.
   */
  public static function finishBatch($success, $results, $operations) {
    if ($success) {
      \Drupal::messenger()->addStatus(t('Queue processed successfully.'));
    }
    else {
      \Drupal::messenger()->addError(t('There was a problem processing the queue.'));
    }
  }

}
