<?php

namespace Drupal\node_health;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;

/**
 * Checks and stores node table sizes.
 */
class NodeHealthSizeCron {
  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;
  
  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  public function __construct(Connection $database, ConfigFactoryInterface $config_factory) {
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * Runs the checker and stores sizes.
   */
  public function run(): void {
    $tables = \Drupal::database()->query("SHOW TABLE STATUS LIKE 'node%'")->fetchAllAssoc('Name');
    // Now returns actual table names.
    $tables_used = $this->getTables();
    $timestamp = time();
    $tables_arr = [];

    foreach ($tables as $name => $info) {
      $size = (int) $info->Data_length + (int) $info->Index_length;
      $count = (int) \Drupal::database()->query("SELECT COUNT(*) FROM {{$name}}")->fetchField();

      if ($name !== 'node_health_sizes' && $name !== 'node_health_table_names' && $name != 'db') {
        $table_id = $this->getOrCreateTableId($name);

        \Drupal::database()->insert('node_health_sizes')
          ->fields([
            'table_id' => $table_id,
            'size' => $size,
            'count' => $count,
            'timestamp' => $timestamp,
          ])->execute();

        $tables_arr[] = $name;
      }
    }

    // Handle tables that existed before but are now missing.
    $diff = array_diff($tables_used, $tables_arr);
    if ($diff) {
      foreach ($diff as $table_name) {
        $table_id = $this->getOrCreateTableId($table_name);
        if ($table_name !== 'node_health_sizes' && $table_name !== 'node_health_table_names' && $table_name != 'db') {
          \Drupal::database()->insert('node_health_sizes')
            ->fields([
              'table_id' => $table_id,
              'size' => 0,
              'count' => 0,
              'timestamp' => $timestamp,
            ])->execute();
        }
      }
    }

    // Log full DB size as a pseudo-table "db".
    $query = \Drupal::database()->query("
      SELECT ROUND(SUM(data_length + index_length), 2) AS size_mb 
      FROM information_schema.tables 
      WHERE table_schema = :schema ",
      [':schema' => \Drupal::database()->getConnectionOptions()['database']]
    );
    $size = $query->fetchField();
    $table_id = $this->getOrCreateTableId('db');

    $total_rows = \Drupal::database()->query("
      SELECT SUM(TABLE_ROWS)
      FROM information_schema.tables
      WHERE table_schema = :schema",
      [':schema' => \Drupal::database()->getConnectionOptions()['database']]
    )->fetchField();

    $id = \Drupal::database()->insert('node_health_sizes')
      ->fields([
        'table_id' => $table_id,
        'size' => $size,
        'count' => $total_rows,
        'timestamp' => $timestamp,
      ])->execute();
  }

  /**
   * Lists all size records with table names.
   */
  public function list() {
    $query = \Drupal::database()->select('node_health_sizes', 'nrh')
      ->fields('nrh', ['size', 'timestamp'])
      ->fields('t', ['table_name'])
      ->orderBy('timestamp', 'ASC');
    $query->innerJoin('node_health_table_names', 't', 'nrh.table_id = t.id');

    return $query->execute()->fetchAll();
  }

  /**
   * Get distinct node table names used.
   */
  private function getTables() {
    $query = \Drupal::database()->select('node_health_sizes', 'nrh');
    $query->innerJoin('node_health_table_names', 't', 'nrh.table_id = t.id');
    $query->addField('t', 'table_name');

    $result = $query->execute()->fetchCol();

    $result = array_unique($result);

    return $result;
  }

  /**
   * Get or create table_id by table name.
   */
  private function getOrCreateTableId(string $table_name): int {
    $id = \Drupal::database()->query("SELECT id FROM {node_health_table_names} WHERE table_name = :name", [
      ':name' => $table_name,
    ])->fetchField();

    if (!$id) {
      $id = \Drupal::database()->insert('node_health_table_names')
        ->fields(['table_name' => $table_name])
        ->execute();
    }

    return (int) $id;
  }

  /**
   * Get total size of all module tables.
   */
  public function getModuleTablesSize() {
    $query = \Drupal::database()->query("
      SELECT ROUND(SUM(data_length + index_length), 2) AS size_mb 
      FROM information_schema.tables 
      WHERE table_schema = :schema ",
      [':schema' => \Drupal::database()->getConnectionOptions()['database']]
    );
    $size = $query->fetchField();
    return $size;
  }

}
