<?php

namespace Drupal\node_in_menu\Plugin\views\filter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filters nodes by whether they are referenced in a menu.
 */
#[ViewsFilter('node_in_menu')]
class NodeInMenu extends FilterPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a NodeInMenu filter plugin.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $menu = $this->value;
    $operator = $this->operator;

    // Ensure the base table is joined and get its alias.
    $table_alias = $this->ensureMyTable();

    // Start with content-based menu links.
    $expression = $this->getContentBasedMenuCondition($table_alias);

    // Add field-based menu links if the table exists.
    if ($this->menuTreeTableExists()) {
      $field_based_part = $this->getFieldBasedMenuCondition($table_alias);
      $expression = "($expression OR $field_based_part)";
    }

    // If operator is 'not in', negate the condition.
    if ($operator === 'not in') {
      $expression = "NOT ($expression)";
    }

    $this->query->addWhereExpression(
      $this->options['group'],
      $expression,
      [':menu_name' => $menu]
    );
  }

  /**
   * Gets the SQL condition for content-based menu links.
   *
   * This method detects nodes that are manually added to menus through the
   * admin interface (not through node fields). These menu items are stored in
   * the menu_link_content_data table with URIs like 'internal:/node/123' or
   * 'entity:node/456'.
   *
   * The query works by:
   * 1. Finding menu links in the specified menu that are enabled
   * 2. Filtering for only node-related URIs (internal:/node/ or entity:node/)
   * 3. Extracting the node ID from the URI using SUBSTRING_INDEX
   * 4. Converting the extracted string to an unsigned integer
   * 5. Checking if the current node's ID is in that list
   *
   * Example URIs and extracted IDs:
   * - 'internal:/node/123' → extracts '123' → node ID 123
   * - 'entity:node/456' → extracts '456' → node ID 456
   * - 'internal:/taxonomy/term/789' → filtered out (not a node)
   *
   * @param string $table_alias
   *   The alias for the base table (e.g., 'node_field_data').
   *
   * @return string
   *   The SQL condition for content-based menu links.
   */
  protected function getContentBasedMenuCondition($table_alias) {
    return "$table_alias.nid IN (
      SELECT CAST(SUBSTRING_INDEX(mlc.link__uri, '/', -1) AS UNSIGNED)
      FROM {menu_link_content_data} mlc
      WHERE mlc.menu_name = :menu_name
        AND mlc.enabled = 1
        AND (mlc.link__uri LIKE 'internal:/node/%' OR mlc.link__uri LIKE 'entity:node/%')
    )";
  }

  /**
   * Gets the SQL condition for field-based menu links.
   *
   * This method detects nodes that are added to menus through their menu link
   * field (e.g., a "Menu link" field on the node edit form). These menu items
   * are stored in the menu_tree table with route_name 'entity.node.canonical'
   * and serialized route parameters.
   *
   * The query works by:
   * 1. Finding menu items in the specified menu that are enabled
   * 2. Filtering for only node canonical routes (entity.node.canonical)
   * 3. Checking if the route parameters contain the current node's ID
   * 4. Using EXISTS for performance (stops searching once a match is found)
   *
   * Route parameters format:
   * The route_parameters field contains serialized PHP data like:
   * 'a:1:{s:4:"node";s:5:"19086";}'.'
   *
   * The LIKE pattern '%\"19086\"%' searches for the quoted node ID within
   * the serialized data, ensuring we find the exact node ID match.
   *
   * @param string $table_alias
   *   The alias for the base table (e.g., 'node_field_data').
   *
   * @return string
   *   The SQL condition for field-based menu links.
   */
  protected function getFieldBasedMenuCondition($table_alias) {
    return "EXISTS (
      SELECT 1 FROM {menu_tree} mt
      WHERE mt.menu_name = :menu_name
        AND mt.enabled = 1
        AND mt.route_name = 'entity.node.canonical'
        AND mt.route_parameters LIKE CONCAT('%\"', CAST($table_alias.nid AS CHAR), '\"%')
    )";
  }

  /**
   * Checks if the menu_tree table exists in the database.
   *
   * @return bool
   *   TRUE if the menu_tree table exists, FALSE otherwise.
   */
  protected function menuTreeTableExists() {
    try {
      $connection = $this->query->getConnection();
      $schema = $connection->schema();
      return $schema->tableExists('menu_tree');
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function valueForm(&$form, FormStateInterface $form_state) {
    $form['operator'] = [
      '#type' => 'select',
      '#title' => $this->t('Operator'),
      '#options' => [
        'in' => $this->t('is in'),
        'not in' => $this->t('is not in'),
      ],
      '#default_value' => $this->operator,
    ];

    $form['value'] = [
      '#type' => 'select',
      '#title' => $this->t('Menu'),
      '#options' => $this->getMenuOptions(),
      '#default_value' => $this->value,
    ];
  }

  /**
   * Returns an array of available menus.
   *
   * @return array
   *   An array of menu machine names => labels.
   */
  protected function getMenuOptions() {
    $menus = $this->entityTypeManager->getStorage('menu')->loadMultiple();
    $options = [];
    foreach ($menus as $menu) {
      $options[$menu->id()] = $menu->label();
    }
    return $options;
  }

}
