<?php

namespace Drupal\node_link_report\Plugin\Block;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node_link_report\Service\LinkChecker;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Node Report Block.
 *
 * @Block(
 *   id = "node_link_report_block",
 *   admin_label = @Translation("Node Link Report"),
 * )
 */
class NodeLinkReport extends BlockBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * Link checking service.
   *
   * @var \Drupal\node_link_report\Service\LinkChecker
   */
  protected $linkChecker;

  /**
   * The scheme and host of the current site.
   *
   * @var string
   */
  private $currentSchemeAndHost;

  /**
   * Keyed by bad, good, skipped, and unpublished.
   *
   * @var array
   */
  private $links;

  /**
   * Configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private $configFactory;

  /**
   * Configuration settings object for node_link_report module.
   *
   * @var object
   */
  private $nlrConfig;

  /**
   * The node object for the current node.
   *
   * @var \Drupal\node\NodeInterface
   */
  private $node = NULL;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private $entityTypeManager;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  private $routeMatch;

  /**
   * The cache backend interface.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  private $cacheBackend;

  /**
   * The HTTP request object.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  private $requestStack;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  private $renderer;

  /**
   * Node Link Report block construct.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param array $configuration
   *   The plugin configuration, i.e. an array with configuration values keyed
   *   by configuration option name.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend_interface
   *   The cache backend interface.
   * @param \Drupal\Core\Renderer\RendererInterface $renderer_interface
   *   The renderer service.
   * @param \Drupal\node_link_report\Service\LinkChecker $link_checker
   *   Link checking service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, array $configuration, $plugin_id, $plugin_definition, RouteMatchInterface $route_match, CacheBackendInterface $cache_backend_interface, RendererInterface $renderer_interface, LinkChecker $link_checker) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->nlrConfig = $this->configFactory->get('node_link_report.settings');
    $this->routeMatch = $route_match;
    $this->cacheBackend = $cache_backend_interface;
    $this->renderer = $renderer_interface;
    $this->linkChecker = $link_checker;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('config.factory'),
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match'),
      $container->get('cache.default'),
      $container->get('renderer'),
      $container->get('node_link_report.link_checker')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $nid = $this->linkChecker->getNode()->id();
    $cid = "node_link_report_{$nid}";
    // Check for cached data.
    $cached_data = $this->cacheBackend->get($cid);
    if (!empty($cached_data) && (!$this->isPreview())) {
      $renderable = $cached_data->data;
    }
    else {
      // Cache unavailable so generate the data.
      $this->linkChecker->checkLinks();
      $renderable = [
        '#theme' => 'node_link_report_block',
        '#node_id' => $nid,
        '#bad_links' => $this->linkChecker->links['bad'],
        '#display_inaccessible_links' => $this->nlrConfig->get('enable_reporting_inaccessible_links'),
        '#inaccessible_links' => $this->linkChecker->links['inaccessible'],
        '#accessibility_guidance_url'  => $this->nlrConfig->get('accessibility_guidance_url'),
        '#accessibility_guidance_link' => $this->nlrConfig->get('accessibility_guidance_link'),
        '#display_good_links' => $this->nlrConfig->get('enable_reporting_good_links'),
        '#good_links' => $this->linkChecker->links['good'],
        '#display_skipped_links' => $this->nlrConfig->get('enable_reporting_skipped_links'),
        '#skipped_links' => $this->linkChecker->links['skipped'],
        '#redirected_links' => $this->linkChecker->links['redirected'],
        '#unpublished_links' => $this->linkChecker->links['unpublished'],
        '#attached' => [
          'library' => [
            'node_link_report/node-link-report',
          ],
        ],
      ];

      if (!$this->isPreview()) {
        // This is not a preview, so cache the data. for 24 hours.
        $expire_time = time() + 24 * 60 * 60;
        $tags = [
          'node_link_report',
          "node:{$nid}",
        ];
        $this->cacheBackend->set($cid, $renderable, $expire_time, $tags);
      }
    }

    return [
      '#markup' => $this->renderer->render($renderable),
    ];
  }

  /**
   * Checks if this is node preview.
   *
   * @return bool
   *   TRUE if it is node preview.  FALSE otherwise.
   */
  private function isPreview() {
    $route_name = $this->routeMatch->getRouteName();
    return ($route_name === "entity.node.preview") ? TRUE : FALSE;
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account) {
    // Check to see if we are on node/NID. node/NID/edit or node preview.
    $route_name = $this->routeMatch->getRouteName();
    $allowed_routes = [];
    if ($this->nlrConfig->get('display_on_node_view')) {
      $allowed_routes[] = 'entity.node.canonical';
    }
    if ($this->nlrConfig->get('display_on_node_edit')) {
      $allowed_routes[] = 'entity.node.edit_form';
    }
    if ($this->nlrConfig->get('display_on_node_preview')) {
      $allowed_routes[] = 'entity.node.preview';
    }

    if (in_array($route_name, $allowed_routes)) {
      return AccessResult::allowedIfHasPermission($account, 'view node link report');
    }
    else {
      // Not on the right path, hide the block.
      return AccessResult::forbiddenIf(TRUE);
    }

  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

}
