<?php

namespace Drupal\node_role_variants\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines a Node Role Variant Set configuration entity.
 *
 * @ConfigEntityType(
 *   id = "node_role_variant_set",
 *   label = @Translation("Node Role Variant Set"),
 *   label_collection = @Translation("Node Role Variant Sets"),
 *   label_singular = @Translation("node role variant set"),
 *   label_plural = @Translation("node role variant sets"),
 *   handlers = {
 *     "list_builder" = "Drupal\node_role_variants\NodeRoleVariantSetListBuilder",
 *   },
 *   config_prefix = "variant_set",
 *   admin_permission = "administer node role variants",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "primary_uuid",
 *     "shared_path",
 *     "variants",
 *   }
 * )
 */
class NodeRoleVariantSet extends ConfigEntityBase implements NodeRoleVariantSetInterface {

  /**
   * The variant set ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The variant set label.
   *
   * @var string
   */
  protected $label;

  /**
   * The primary node UUID.
   *
   * @var string
   */
  protected $primary_uuid;

  /**
   * Whether to share path alias across variants.
   *
   * @var bool
   */
  protected $shared_path = TRUE;

  /**
   * The role variants.
   *
   * @var array
   */
  protected $variants = [];

  /**
   * {@inheritdoc}
   */
  public function getPrimaryUuid() {
    return $this->primary_uuid;
  }

  /**
   * {@inheritdoc}
   */
  public function setPrimaryUuid($uuid) {
    $this->primary_uuid = $uuid;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getSharedPath() {
    return $this->shared_path;
  }

  /**
   * {@inheritdoc}
   */
  public function setSharedPath($shared_path) {
    $this->shared_path = $shared_path;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getVariants() {
    return $this->variants;
  }

  /**
   * {@inheritdoc}
   */
  public function setVariants(array $variants) {
    $this->variants = $variants;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function addVariant($role_id, $variant_uuid, $weight = 0) {
    $this->variants[$role_id] = [
      'role_id' => $role_id,
      'variant_uuid' => $variant_uuid,
      'weight' => (int) $weight,
    ];
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function removeVariant($role_id) {
    unset($this->variants[$role_id]);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getVariant($role_id) {
    return $this->variants[$role_id] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function hasVariant($role_id) {
    return isset($this->variants[$role_id]);
  }

  /**
   * {@inheritdoc}
   */
  public function getVariantUuidForRole($role_id) {
    return $this->variants[$role_id]['variant_uuid'] ?? NULL;
  }

}
