<?php

namespace Drupal\node_role_variants\EventSubscriber;

use Drupal\Core\Cache\CacheableRedirectResponse;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\ResettableStackedRouteMatchInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Drupal\node_role_variants\Service\NodeRoleVariantsManager;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for handling role-based node redirects.
 */
class NodeViewSubscriber implements EventSubscriberInterface {

  /**
   * The role variants manager.
   *
   * @var \Drupal\node_role_variants\Service\NodeRoleVariantsManager
   */
  protected $variantsManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current route match service (resettable).
   *
   * @var \Drupal\Core\Routing\ResettableStackedRouteMatchInterface
   */
  protected $currentRouteMatch;

  /**
   * Constructs a NodeViewSubscriber object.
   *
   * @param \Drupal\node_role_variants\Service\NodeRoleVariantsManager $variants_manager
   *   The role variants manager.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Routing\ResettableStackedRouteMatchInterface $current_route_match
   *   The current route match service (resettable).
   */
  public function __construct(
    NodeRoleVariantsManager $variants_manager,
    AccountInterface $current_user,
    RouteMatchInterface $route_match,
    RequestStack $request_stack,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    ResettableStackedRouteMatchInterface $current_route_match,
  ) {
    $this->variantsManager = $variants_manager;
    $this->currentUser = $current_user;
    $this->routeMatch = $route_match;
    $this->requestStack = $request_stack;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->currentRouteMatch = $current_route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    // Priority 28 runs after routing (32) but before Dynamic Page Cache (27).
    // This ensures route parameters are available but we can swap node before caching.
    return [
      KernelEvents::REQUEST => ['onRequest', 28],
    ];
  }

  /**
   * Handles the request event for role-based redirects.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onRequest(RequestEvent $event) {
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();

    // Skip if admin preview mode is requested.
    if ($request->query->get('no-role-redirect') === '1') {
      return;
    }

    // Skip API/AJAX requests.
    if ($this->isApiRequest($request)) {
      return;
    }

    // Get the route name from the request attributes.
    $route_name = $request->attributes->get('_route');

    // Get the node - handle both canonical and front page routes.
    $node = NULL;
    $is_node_canonical_route = FALSE;

    if ($route_name === 'entity.node.canonical') {
      $node = $request->attributes->get('node');
      $is_node_canonical_route = TRUE;
    }
    elseif ($route_name === '<front>' || $route_name === 'view.frontpage.page_1') {
      // Check if front page is a node.
      $node = $this->getFrontPageNode();
      // Front page via Views cannot use node swapping, only redirects work.
      // The route is NOT a canonical node route in this case.
    }

    if (!$node instanceof NodeInterface) {
      return;
    }

    // Check if role variants are enabled for this content type.
    if (!$this->variantsManager->isEnabledForNode($node)) {
      return;
    }

    // Check if shared path is enabled for this specific node/variant group.
    // Note: shared_path swapping only works on node canonical routes.
    // For Views-based front pages, we must redirect regardless of this setting.
    $shared_path = $this->variantsManager->getSharedPathForNode($node->uuid());
    if (!$is_node_canonical_route) {
      // Force redirect for non-canonical routes (Views, etc.) since
      // node swapping won't work with Views rendering.
      $shared_path = FALSE;
    }

    // Get the appropriate variant for the current user.
    $variant = $this->variantsManager->getVariantForUser($node, $this->currentUser);

    if ($variant instanceof NodeInterface) {
      // Check if user has access to the variant.
      if (!$variant->access('view', $this->currentUser)) {
        return;
      }

      if ($shared_path) {
        // Shared path: Swap the node in the request, don't redirect.
        // This keeps the URL the same but serves the variant content.
        $request->attributes->set('node', $variant);

        // Update _entity attribute which EntityViewController uses for rendering.
        // This ensures the variant is rendered with the same view mode as the primary.
        $request->attributes->set('_entity', $variant);

        // Explicitly set view_mode to 'full' to match the primary node's rendering.
        // NodeViewController defaults to 'full', but we set it explicitly to ensure
        // consistency regardless of how the controller resolves the parameter.
        $request->attributes->set('view_mode', 'full');

        // Also update the route parameters for proper rendering.
        if ($request->attributes->has('_raw_variables')) {
          $raw_variables = $request->attributes->get('_raw_variables');
          if ($raw_variables) {
            $raw_variables->set('node', $variant->id());
          }
        }

        // CRITICAL: Reset the cached RouteMatch so that node_is_page() returns
        // TRUE for the swapped node. The CurrentRouteMatch service caches the
        // RouteMatch in an SplObjectStorage. By resetting it, the next call to
        // \Drupal::routeMatch()->getParameter('node') will use
        // RouteMatch::createFromRequest() which reads from $request->attributes
        // and will get our swapped node. Without this, the 'page' variable in
        // template_preprocess_node() would be FALSE, causing the node title to
        // be rendered when it shouldn't be.
        $this->currentRouteMatch->resetRouteMatch();
      }
      else {
        // Different paths: Check if we need to redirect.
        $current_uuid = $node->uuid();

        // Check if user is on a variant that's assigned to one of their roles.
        $variant_info = $this->variantsManager->getVariantInfo($current_uuid);
        if ($variant_info) {
          $user_roles = $this->currentUser->getRoles();
          if (\in_array($variant_info['role_id'], $user_roles, TRUE)) {
            // User is on a variant assigned to their role, no redirect needed.
            return;
          }
        }

        // Build the redirect URL.
        $url = Url::fromRoute('entity.node.canonical', ['node' => $variant->id()]);

        // Preserve query parameters except the redirect bypass.
        $query = $request->query->all();
        unset($query['no-role-redirect']);
        if (!empty($query)) {
          $url->setOption('query', $query);
        }

        // Create a cacheable redirect response (302 temporary).
        $response = new CacheableRedirectResponse($url->toString(), 302);

        // Add cache contexts for user roles.
        $response->getCacheableMetadata()
          ->addCacheContexts(['user.roles', 'url.query_args:no-role-redirect'])
          ->addCacheTags(['node:' . $node->id(), 'node:' . $variant->id()]);

        $event->setResponse($response);
      }
    }
  }

  /**
   * Get the front page node if the front page is set to a node.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The front page node or NULL.
   */
  protected function getFrontPageNode() {
    $front_page = $this->configFactory->get('system.site')->get('page.front');

    if ($front_page && preg_match('#^/node/(\d+)$#', $front_page, $matches)) {
      $nid = $matches[1];
      return $this->entityTypeManager->getStorage('node')->load($nid);
    }

    return NULL;
  }

  /**
   * Check if the request is an API request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return bool
   *   TRUE if this is an API request.
   */
  protected function isApiRequest($request) {
    // Check for common API request indicators.
    $accept = $request->headers->get('Accept', '');
    $content_type = $request->headers->get('Content-Type', '');

    // JSON API requests.
    if (strpos($accept, 'application/json') !== FALSE ||
        strpos($accept, 'application/vnd.api+json') !== FALSE ||
        strpos($content_type, 'application/json') !== FALSE) {
      return TRUE;
    }

    // AJAX requests.
    if ($request->isXmlHttpRequest()) {
      return TRUE;
    }

    // Check for _format parameter.
    $format = $request->query->get('_format');
    if ($format && $format !== 'html') {
      return TRUE;
    }

    return FALSE;
  }

}
