<?php

namespace Drupal\node_role_variants\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Drupal\node_role_variants\Service\NodeRoleVariantsManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for cloning a node as a role variant.
 */
class CloneAsVariantForm extends FormBase {

  /**
   * The role variants manager.
   *
   * @var \Drupal\node_role_variants\Service\NodeRoleVariantsManager
   */
  protected $variantsManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a CloneAsVariantForm object.
   *
   * @param \Drupal\node_role_variants\Service\NodeRoleVariantsManager $variants_manager
   *   The role variants manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    NodeRoleVariantsManager $variants_manager,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->variantsManager = $variants_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('node_role_variants.manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_role_variants_clone_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {
    if (!$node) {
      return $form;
    }

    $form_state->set('node', $node);
    $uuid = $node->uuid();
    $nid = (int) $node->id();

    // Check if this node is already a variant.
    $variant_info = $this->variantsManager->getVariantInfo($uuid);
    if ($variant_info) {
      $this->messenger()->addWarning($this->t('This node is already a variant of another node. You cannot clone a variant as a new variant.'));
      return $this->redirect('node_role_variants.node_variants', ['node' => $nid]);
    }

    // Get all roles.
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $role_options = [];
    foreach ($roles as $role) {
      $role_options[$role->id()] = $role->label();
    }

    // Get existing variants and exclude used roles.
    $variants = $this->variantsManager->getVariants($uuid);
    $used_roles = array_keys($variants);
    $available_roles = array_diff_key($role_options, array_flip($used_roles));

    if (empty($available_roles)) {
      $this->messenger()->addWarning($this->t('All roles already have variants configured.'));
      return $this->redirect('node_role_variants.node_variants', ['node' => $nid]);
    }

    $form['info'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--status']],
      'message' => [
        '#markup' => $this->t('This will create a copy of <strong>@title</strong> and set it up as a role variant. The cloned node will be unpublished by default so you can review and customize it before publishing.', [
          '@title' => $node->label(),
        ]),
      ],
    ];

    $form['role'] = [
      '#type' => 'select',
      '#title' => $this->t('Target role'),
      '#description' => $this->t('Select the role that will see this cloned variant.'),
      '#options' => $available_roles,
      '#empty_option' => $this->t('- Select a role -'),
      '#required' => TRUE,
    ];

    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Variant title'),
      '#description' => $this->t('The title for the cloned variant node. This helps distinguish it from the original in the admin interface.'),
      '#default_value' => $node->label() . ' (' . $this->t('Variant') . ')',
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['clone_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Clone options'),
      '#open' => TRUE,
    ];

    $form['clone_options']['clone_paragraphs'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Clone paragraph references'),
      '#description' => $this->t('If enabled, paragraph entities referenced by this node will be duplicated. If disabled, the clone will reference the same paragraph entities as the original.'),
      '#default_value' => TRUE,
    ];

    $form['clone_options']['published'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Publish the cloned variant'),
      '#description' => $this->t('If disabled, the clone will be created as unpublished so you can review it first.'),
      '#default_value' => FALSE,
    ];

    // Variant set configuration (only for new variant sets).
    $is_new_variant_set = empty($variants) && !$this->variantsManager->isPrimaryNode($uuid);
    if ($is_new_variant_set) {
      $form['variant_set'] = [
        '#type' => 'details',
        '#title' => $this->t('Variant set configuration'),
        '#open' => FALSE,
      ];

      $form['variant_set']['variant_set_label'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Variant set label'),
        '#description' => $this->t('A human-readable label for this variant set.'),
        '#default_value' => $node->label(),
      ];

      $form['variant_set']['variant_set_id'] = [
        '#type' => 'machine_name',
        '#title' => $this->t('Variant set ID'),
        '#description' => $this->t('A unique machine name for this variant set.'),
        '#default_value' => '',
        '#machine_name' => [
          'exists' => [$this, 'variantSetIdExists'],
          'source' => ['variant_set', 'variant_set_label'],
          'replace_pattern' => '[^a-z0-9_]+',
          'replace' => '_',
        ],
        '#required' => FALSE,
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Clone as variant'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('node_role_variants.node_variants', ['node' => $nid]),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\node\NodeInterface $node */
    $node = $form_state->get('node');
    $uuid = $node->uuid();
    $role = $form_state->getValue('role');
    $title = $form_state->getValue('title');
    $clone_paragraphs = (bool) $form_state->getValue('clone_paragraphs');
    $published = (bool) $form_state->getValue('published');

    try {
      // Clone the node.
      $clone = $this->cloneNode($node, $title, $clone_paragraphs, $published);

      if (!$clone) {
        $this->messenger()->addError($this->t('Failed to create the cloned node.'));
        return;
      }

      // Get variant set configuration.
      $variant_set_id = $form_state->getValue('variant_set_id') ?: NULL;
      $variant_set_label = $form_state->getValue('variant_set_label') ?: $node->label();

      // Get the next weight.
      $variants = $this->variantsManager->getVariants($uuid);
      $max_weight = 0;
      foreach ($variants as $variant) {
        if ($variant['weight'] > $max_weight) {
          $max_weight = $variant['weight'];
        }
      }

      // Add as variant using UUIDs.
      $this->variantsManager->addVariant(
        $uuid,
        $clone->uuid(),
        $role,
        $max_weight + 1,
        $variant_set_id,
        $variant_set_label
      );

      $role_entity = $this->entityTypeManager->getStorage('user_role')->load($role);
      $role_label = $role_entity ? $role_entity->label() : $role;

      $this->messenger()->addStatus($this->t('Created variant "@title" for the "@role" role.', [
        '@title' => $clone->label(),
        '@role' => $role_label,
      ]));

      // Redirect to edit the new clone.
      $form_state->setRedirectUrl($clone->toUrl('edit-form'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('An error occurred while cloning: @message', [
        '@message' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Clones a node entity.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to clone.
   * @param string $title
   *   The title for the cloned node.
   * @param bool $clone_paragraphs
   *   Whether to clone paragraph references.
   * @param bool $published
   *   Whether the clone should be published.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The cloned node, or NULL on failure.
   */
  protected function cloneNode(NodeInterface $node, string $title, bool $clone_paragraphs, bool $published): ?NodeInterface {
    /** @var \Drupal\node\NodeInterface $clone */
    $clone = $node->createDuplicate();

    // Set the new title.
    $clone->setTitle($title);

    // Set published status.
    if ($published) {
      $clone->setPublished();
    }
    else {
      $clone->setUnpublished();
    }

    // Reset some fields that shouldn't be cloned.
    $clone->setCreatedTime(\Drupal::time()->getRequestTime());
    $clone->setChangedTime(\Drupal::time()->getRequestTime());
    $clone->setRevisionCreationTime(\Drupal::time()->getRequestTime());
    $clone->setOwnerId(\Drupal::currentUser()->id());
    $clone->setRevisionUserId(\Drupal::currentUser()->id());

    // Handle paragraph fields if requested.
    if ($clone_paragraphs) {
      $this->cloneParagraphFields($clone);
    }

    // Save the clone.
    $clone->save();

    return $clone;
  }

  /**
   * Clones paragraph field references on a node.
   *
   * @param \Drupal\node\NodeInterface $clone
   *   The cloned node (not yet saved).
   */
  protected function cloneParagraphFields(NodeInterface $clone): void {
    // Check if the Paragraphs module is enabled.
    if (!\Drupal::moduleHandler()->moduleExists('paragraphs')) {
      return;
    }

    $field_definitions = $clone->getFieldDefinitions();

    foreach ($field_definitions as $field_name => $field_definition) {
      $field_type = $field_definition->getType();

      // Check for entity_reference_revisions fields (used by Paragraphs).
      if ($field_type === 'entity_reference_revisions') {
        $settings = $field_definition->getSettings();
        $target_type = $settings['target_type'] ?? '';

        if ($target_type === 'paragraph') {
          $this->cloneParagraphField($clone, $field_name);
        }
      }
    }
  }

  /**
   * Clones a single paragraph field.
   *
   * @param \Drupal\node\NodeInterface $clone
   *   The cloned node.
   * @param string $field_name
   *   The field name.
   */
  protected function cloneParagraphField(NodeInterface $clone, string $field_name): void {
    if (!$clone->hasField($field_name)) {
      return;
    }

    $field = $clone->get($field_name);
    $new_paragraphs = [];

    foreach ($field as $item) {
      $paragraph = $item->entity;
      if ($paragraph) {
        $cloned_paragraph = $this->cloneParagraphRecursive($paragraph);
        if ($cloned_paragraph) {
          $new_paragraphs[] = $cloned_paragraph;
        }
      }
    }

    $clone->set($field_name, $new_paragraphs);
  }

  /**
   * Recursively clones a paragraph and its nested paragraphs.
   *
   * @param object $paragraph
   *   The paragraph to clone.
   *
   * @return object|null
   *   The cloned paragraph.
   */
  protected function cloneParagraphRecursive($paragraph) {
    if (!$paragraph) {
      return NULL;
    }

    $clone = $paragraph->createDuplicate();

    // Process nested paragraph fields.
    $field_definitions = $clone->getFieldDefinitions();
    foreach ($field_definitions as $field_name => $field_definition) {
      if ($field_definition->getType() === 'entity_reference_revisions') {
        $settings = $field_definition->getSettings();
        if (($settings['target_type'] ?? '') === 'paragraph') {
          $nested_paragraphs = [];
          foreach ($clone->get($field_name) as $item) {
            if ($item->entity) {
              $nested_clone = $this->cloneParagraphRecursive($item->entity);
              if ($nested_clone) {
                $nested_paragraphs[] = $nested_clone;
              }
            }
          }
          $clone->set($field_name, $nested_paragraphs);
        }
      }
    }

    return $clone;
  }

  /**
   * Checks if a variant set ID already exists.
   *
   * @param string $id
   *   The variant set ID to check.
   *
   * @return bool
   *   TRUE if the ID exists, FALSE otherwise.
   */
  public function variantSetIdExists($id) {
    $variant_set = $this->entityTypeManager->getStorage('node_role_variant_set')->load($id);
    return $variant_set !== NULL;
  }

}
