<?php

namespace Drupal\node_role_variants\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Drupal\node_role_variants\Service\NodeRoleVariantsManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for removing a role variant.
 */
class RemoveVariantConfirmForm extends ConfirmFormBase {

  /**
   * The role variants manager.
   *
   * @var \Drupal\node_role_variants\Service\NodeRoleVariantsManager
   */
  protected $variantsManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node;

  /**
   * The role ID.
   *
   * @var string
   */
  protected $roleId;

  /**
   * Constructs a RemoveVariantConfirmForm object.
   *
   * @param \Drupal\node_role_variants\Service\NodeRoleVariantsManager $variants_manager
   *   The role variants manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    NodeRoleVariantsManager $variants_manager,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    $this->variantsManager = $variants_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('node_role_variants.manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_role_variants_remove_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    $role = $this->entityTypeManager->getStorage('user_role')->load($this->roleId);
    return $this->t('Are you sure you want to remove the @role variant from %title?', [
      '@role' => $role ? $role->label() : $this->roleId,
      '%title' => $this->node->label(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This will remove the role variant mapping. The variant node itself will not be deleted.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return Url::fromRoute('node_role_variants.node_variants', ['node' => $this->node->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL, $role_id = NULL) {
    $this->node = $node;
    $this->roleId = $role_id;

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->variantsManager->removeVariant($this->node->uuid(), $this->roleId);

    $role = $this->entityTypeManager->getStorage('user_role')->load($this->roleId);
    $this->messenger()->addStatus($this->t('The @role variant has been removed.', [
      '@role' => $role ? $role->label() : $this->roleId,
    ]));

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
