<?php

namespace Drupal\node_role_variants\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Drupal\node_role_variants\Entity\NodeRoleVariantSetInterface;

/**
 * Service for managing role-based node variants.
 */
class NodeRoleVariantsManager {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Constructs a NodeRoleVariantsManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    AccountInterface $current_user,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
  }

  /**
   * Get the variant set storage.
   *
   * @return \Drupal\Core\Entity\EntityStorageInterface
   *   The variant set storage.
   */
  protected function getStorage() {
    return $this->entityTypeManager->getStorage('node_role_variant_set');
  }

  /**
   * Get the node storage.
   *
   * @return \Drupal\Core\Entity\EntityStorageInterface
   *   The node storage.
   */
  protected function getNodeStorage() {
    return $this->entityTypeManager->getStorage('node');
  }

  /**
   * Load a node by UUID.
   *
   * @param string $uuid
   *   The node UUID.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The node or NULL if not found.
   */
  public function loadNodeByUuid(string $uuid): ?NodeInterface {
    $nodes = $this->getNodeStorage()->loadByProperties(['uuid' => $uuid]);
    return $nodes ? reset($nodes) : NULL;
  }

  /**
   * Get the appropriate variant node for a user based on their roles.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node being viewed (primary or variant).
   * @param \Drupal\Core\Session\AccountInterface|null $user
   *   The user to check. Defaults to current user.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The appropriate variant node, or NULL if no redirect needed.
   */
  public function getVariantForUser(NodeInterface $node, AccountInterface $user = NULL) {
    if ($user === NULL) {
      $user = $this->currentUser;
    }

    $uuid = $node->uuid();

    // Get the variant set for this node (either as primary or variant).
    $variant_set = $this->getVariantSetForNode($uuid);
    if (!$variant_set) {
      return NULL;
    }

    $primary_uuid = $variant_set->getPrimaryUuid();
    $variants = $variant_set->getVariants();

    // Get user's roles.
    $user_roles = $user->getRoles();

    // Sort variants by weight and find the best match.
    uasort($variants, function ($a, $b) {
      return ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0);
    });

    foreach ($variants as $variant) {
      if (\in_array($variant['role_id'], $user_roles, TRUE)) {
        $variant_uuid = $variant['variant_uuid'];
        // Only return if different from current node.
        if ($variant_uuid !== $uuid) {
          return $this->loadNodeByUuid($variant_uuid);
        }
        // User is already on their correct variant.
        return NULL;
      }
    }

    // No matching variant, return primary node if we're on a variant.
    if ($primary_uuid !== $uuid) {
      return $this->loadNodeByUuid($primary_uuid);
    }

    return NULL;
  }

  /**
   * Get the variant set for a node (either as primary or as a variant).
   *
   * @param string $uuid
   *   The node UUID.
   *
   * @return \Drupal\node_role_variants\Entity\NodeRoleVariantSetInterface|null
   *   The variant set or NULL.
   */
  public function getVariantSetForNode(string $uuid) {
    // First check if this is a primary node.
    $variant_set = $this->getVariantSetByPrimaryUuid($uuid);
    if ($variant_set) {
      return $variant_set;
    }

    // Check if this is a variant node.
    return $this->getVariantSetByVariantUuid($uuid);
  }

  /**
   * Get variant set by primary node UUID.
   *
   * @param string $primary_uuid
   *   The primary node UUID.
   *
   * @return \Drupal\node_role_variants\Entity\NodeRoleVariantSetInterface|null
   *   The variant set or NULL.
   */
  public function getVariantSetByPrimaryUuid(string $primary_uuid) {
    $sets = $this->getStorage()->loadByProperties(['primary_uuid' => $primary_uuid]);
    return $sets ? reset($sets) : NULL;
  }

  /**
   * Get variant set by variant node UUID.
   *
   * @param string $variant_uuid
   *   The variant node UUID.
   *
   * @return \Drupal\node_role_variants\Entity\NodeRoleVariantSetInterface|null
   *   The variant set or NULL.
   */
  public function getVariantSetByVariantUuid(string $variant_uuid) {
    $sets = $this->getStorage()->loadMultiple();
    foreach ($sets as $set) {
      foreach ($set->getVariants() as $variant) {
        if ($variant['variant_uuid'] === $variant_uuid) {
          return $set;
        }
      }
    }
    return NULL;
  }

  /**
   * Get all variants linked to a primary node.
   *
   * @param string $primary_uuid
   *   The primary node UUID.
   *
   * @return array
   *   Array of variant data, keyed by role_id.
   */
  public function getVariants(string $primary_uuid) {
    $variant_set = $this->getVariantSetByPrimaryUuid($primary_uuid);
    if (!$variant_set) {
      return [];
    }

    $variants = $variant_set->getVariants();

    // Sort by weight.
    uasort($variants, function ($a, $b) {
      return ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0);
    });

    return $variants;
  }

  /**
   * Get the primary node UUID for a variant.
   *
   * @param string $variant_uuid
   *   The variant node UUID.
   *
   * @return string|null
   *   The primary node UUID, or NULL if this is not a variant.
   */
  public function getPrimaryNodeUuid(string $variant_uuid) {
    $variant_set = $this->getVariantSetByVariantUuid($variant_uuid);
    return $variant_set ? $variant_set->getPrimaryUuid() : NULL;
  }

  /**
   * Create or get a variant set for a primary node.
   *
   * @param string $primary_uuid
   *   The primary node UUID.
   * @param string $label
   *   The label for the variant set.
   * @param string|null $custom_id
   *   Optional custom ID for the variant set.
   *
   * @return \Drupal\node_role_variants\Entity\NodeRoleVariantSetInterface
   *   The variant set.
   */
  public function getOrCreateVariantSet(string $primary_uuid, string $label = '', ?string $custom_id = NULL) {
    $variant_set = $this->getVariantSetByPrimaryUuid($primary_uuid);
    if ($variant_set) {
      return $variant_set;
    }

    // Create a new variant set.
    $id = $custom_id ?: 'node_' . substr($primary_uuid, 0, 8);
    $variant_set = $this->getStorage()->create([
      'id' => $id,
      'label' => $label ?: 'Variant set for node ' . $primary_uuid,
      'primary_uuid' => $primary_uuid,
      'shared_path' => TRUE,
      'variants' => [],
    ]);

    return $variant_set;
  }

  /**
   * Link a variant node to a primary node for a role.
   *
   * @param string $primary_uuid
   *   The primary node UUID.
   * @param string $variant_uuid
   *   The variant node UUID.
   * @param string $role_id
   *   The role machine name.
   * @param int $weight
   *   Priority weight (lower = higher priority).
   * @param string|null $variant_set_id
   *   Optional custom ID for the variant set (only used when creating new set).
   * @param string|null $variant_set_label
   *   Optional label for the variant set (only used when creating new set).
   *
   * @throws \InvalidArgumentException
   *   If trying to create a circular reference.
   */
  public function addVariant(string $primary_uuid, string $variant_uuid, string $role_id, int $weight = 0, ?string $variant_set_id = NULL, ?string $variant_set_label = NULL) {
    // Prevent circular references.
    if ($primary_uuid === $variant_uuid) {
      throw new \InvalidArgumentException('A node cannot be its own variant.');
    }

    // Check if variant_uuid is already a primary node.
    if ($this->isPrimaryNode($variant_uuid)) {
      throw new \InvalidArgumentException('Cannot use a primary node as a variant. The node already has variants assigned to it.');
    }

    // Check if primary_uuid is already a variant.
    if ($this->isVariantNode($primary_uuid)) {
      throw new \InvalidArgumentException('Cannot use a variant node as a primary. The node is already a variant of another node.');
    }

    // Check if variant_uuid is already a variant of another primary.
    $existing_set = $this->getVariantSetByVariantUuid($variant_uuid);
    if ($existing_set && $existing_set->getPrimaryUuid() !== $primary_uuid) {
      throw new \InvalidArgumentException('This node is already a variant of another node.');
    }

    // Get or create the variant set.
    $node = $this->loadNodeByUuid($primary_uuid);
    $label = $variant_set_label ?: ($node ? $node->label() : 'Node ' . $primary_uuid);
    $variant_set = $this->getOrCreateVariantSet($primary_uuid, $label, $variant_set_id);

    // Add the variant.
    $variant_set->addVariant($role_id, $variant_uuid, $weight);
    $variant_set->save();
  }

  /**
   * Remove a variant link.
   *
   * @param string $primary_uuid
   *   The primary node UUID.
   * @param string $role_id
   *   The role machine name.
   */
  public function removeVariant(string $primary_uuid, string $role_id) {
    $variant_set = $this->getVariantSetByPrimaryUuid($primary_uuid);
    if ($variant_set) {
      $variant_set->removeVariant($role_id);

      // If no more variants, delete the variant set.
      if (empty($variant_set->getVariants())) {
        $variant_set->delete();
      }
      else {
        $variant_set->save();
      }
    }
  }

  /**
   * Check if a node is a primary node (has variants).
   *
   * @param string $uuid
   *   The node UUID.
   *
   * @return bool
   *   TRUE if the node has variants.
   */
  public function isPrimaryNode(string $uuid) {
    return $this->getVariantSetByPrimaryUuid($uuid) !== NULL;
  }

  /**
   * Check if a node is a variant (linked to a primary).
   *
   * @param string $uuid
   *   The node UUID.
   *
   * @return bool
   *   TRUE if the node is a variant.
   */
  public function isVariantNode(string $uuid) {
    return $this->getPrimaryNodeUuid($uuid) !== NULL;
  }

  /**
   * Delete all relationships for a node (when node is deleted).
   *
   * @param string $uuid
   *   The node UUID.
   */
  public function deleteNodeRelationships(string $uuid) {
    // Delete if this node is a primary.
    $variant_set = $this->getVariantSetByPrimaryUuid($uuid);
    if ($variant_set) {
      $variant_set->delete();
      return;
    }

    // Remove from variant sets if this is a variant.
    $sets = $this->getStorage()->loadMultiple();
    foreach ($sets as $set) {
      $variants = $set->getVariants();
      $modified = FALSE;
      foreach ($variants as $role_id => $variant) {
        if ($variant['variant_uuid'] === $uuid) {
          $set->removeVariant($role_id);
          $modified = TRUE;
        }
      }
      if ($modified) {
        if (empty($set->getVariants())) {
          $set->delete();
        }
        else {
          $set->save();
        }
      }
    }
  }

  /**
   * Update weights for variants of a primary node.
   *
   * @param string $primary_uuid
   *   The primary node UUID.
   * @param array $weights
   *   Array of role_id => weight mappings.
   */
  public function updateWeights(string $primary_uuid, array $weights) {
    $variant_set = $this->getVariantSetByPrimaryUuid($primary_uuid);
    if (!$variant_set) {
      return;
    }

    $variants = $variant_set->getVariants();
    foreach ($weights as $role_id => $weight) {
      if (isset($variants[$role_id])) {
        $variants[$role_id]['weight'] = (int) $weight;
      }
    }

    $variant_set->setVariants($variants);
    $variant_set->save();
  }

  /**
   * Check if role variants are enabled for a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to check.
   *
   * @return bool
   *   TRUE if role variants are enabled for this node's type.
   */
  public function isEnabledForNode(NodeInterface $node) {
    return node_role_variants_is_enabled($node->bundle());
  }

  /**
   * Get the primary node data if this node is part of a variant group.
   *
   * @param string $uuid
   *   The node UUID.
   *
   * @return array|null
   *   Array with 'primary_uuid' and 'role_id' if this is a variant, NULL otherwise.
   */
  public function getVariantInfo(string $uuid) {
    $variant_set = $this->getVariantSetByVariantUuid($uuid);
    if (!$variant_set) {
      return NULL;
    }

    foreach ($variant_set->getVariants() as $variant) {
      if ($variant['variant_uuid'] === $uuid) {
        return [
          'primary_uuid' => $variant_set->getPrimaryUuid(),
          'role_id' => $variant['role_id'],
        ];
      }
    }

    return NULL;
  }

  /**
   * Get the shared_path setting for a primary node.
   *
   * @param string $primary_uuid
   *   The primary node UUID.
   *
   * @return bool
   *   TRUE if shared path is enabled (default), FALSE otherwise.
   */
  public function getSharedPath(string $primary_uuid) {
    $variant_set = $this->getVariantSetByPrimaryUuid($primary_uuid);
    return $variant_set ? $variant_set->getSharedPath() : TRUE;
  }

  /**
   * Set the shared_path setting for a primary node.
   *
   * @param string $primary_uuid
   *   The primary node UUID.
   * @param bool $shared_path
   *   Whether to share path alias across variants.
   */
  public function setSharedPath(string $primary_uuid, bool $shared_path) {
    $variant_set = $this->getVariantSetByPrimaryUuid($primary_uuid);
    if ($variant_set) {
      $variant_set->setSharedPath($shared_path);
      $variant_set->save();
    }
  }

  /**
   * Get the shared_path setting for a node (primary or variant).
   *
   * @param string $uuid
   *   The node UUID (can be primary or variant).
   *
   * @return bool
   *   TRUE if shared path is enabled, FALSE otherwise.
   */
  public function getSharedPathForNode(string $uuid) {
    $variant_set = $this->getVariantSetForNode($uuid);
    return $variant_set ? $variant_set->getSharedPath() : TRUE;
  }

}
