# Node Role Variants

Node Role Variants allows you to serve different content to users based on their
roles, similar to how content translations work for multilingual sites. This
enables personalized content experiences where authenticated users, subscribers,
or any custom role can see different versions of the same page.

## Table of Contents

- [Features](#features)
- [Requirements](#requirements)
- [Installation](#installation)
- [Configuration](#configuration)
- [Usage](#usage)
  - [Enabling Role Variants for a Content Type](#enabling-role-variants-for-a-content-type)
  - [Creating Variant Nodes](#creating-variant-nodes)
  - [Configuring Variants](#configuring-variants)
  - [Cloning as a Variant](#cloning-as-a-variant)
  - [Shared Path vs Separate Paths](#shared-path-vs-separate-paths)
  - [Priority Weights](#priority-weights)
- [How It Works](#how-it-works)
- [Template Suggestions](#template-suggestions)
- [Caching](#caching)
- [API Reference](#api-reference)
- [Troubleshooting](#troubleshooting)
- [Maintainers](#maintainers)

## Features

- **Role-Based Content Variants**: Serve different node content based on user
  roles (anonymous, authenticated, custom roles).
- **Shared Path Mode**: Keep the same URL for all variants - users see different
  content at the same path based on their role.
- **Separate Path Mode**: Each variant has its own URL with automatic redirects
  based on user role.
- **Priority Weights**: Control which variant takes precedence when a user has
  multiple roles.
- **Seamless Rendering**: Variants render exactly like canonical nodes with
  proper template selection and view modes.
- **Cache-Aware**: Properly integrates with Drupal's caching system using
  appropriate cache contexts and tags.
- **Admin Preview**: Bypass role redirects with a query parameter for testing.
- **Theme Integration**: Provides template suggestions for role-variant-specific
  theming.
- **Clone as Variant**: Quickly create variant nodes by cloning the primary node,
  with full support for paragraph duplication.

## Requirements

- Drupal 10.x or 11.x
- PHP 8.1 or higher

## Installation

1. Install as you would normally install a contributed Drupal module. Visit
   [Installing Drupal Modules](https://www.drupal.org/docs/extending-drupal/installing-drupal-modules)
   for further information.

2. Enable the module:
   ```bash
   drush en node_role_variants
   ```

## Configuration

### Permissions

Navigate to **Administration > People > Permissions** and configure:

- **Administer node role variants**: Allows users to configure role variants on
  nodes.

## Usage

### Enabling Role Variants for a Content Type

1. Navigate to **Administration > Structure > Content types**.
2. Edit the content type you want to enable role variants for.
3. Expand the **Role variants** section.
4. Check **Enable role-based variants**.
5. Save the content type.

### Creating Variant Nodes

1. Create the primary node that will serve as the default content.
2. Create additional nodes that will serve as variants for specific roles.
3. These variant nodes should be of the same content type (recommended) but can
   be any content type with role variants enabled.

### Configuring Variants

1. Navigate to the primary node's view page.
2. Click the **Role Variants** tab.
3. In the **Add role variant** section:
   - Choose a **Method**: "Reference existing content" or "Clone this node"
   - Select the **Role** this variant applies to.
   - For reference method: Select the **Content** (variant node) to display.
   - For clone method: Enter a **Variant title** for the new node.
4. Click **Add variant**.
5. Optionally configure the **Variant set label** and **ID** in the Settings
   section.
6. Save the configuration.

### Cloning as a Variant

Instead of creating variant nodes manually, you can clone the primary node
directly from the Role Variants interface. This is useful when:

- You want to start with a copy of the primary node and make modifications.
- The primary node has complex paragraph structures you want to preserve.
- You need to quickly create variants for multiple roles.

#### Clone Options

When using the clone method:

- **Variant title**: The title for the cloned node (helps distinguish it in
  admin).
- **Clone paragraph references**: When enabled, all paragraph entities are
  duplicated. When disabled, the clone shares the same paragraph entities as
  the original (changes affect both).
- **Publish the cloned variant**: By default, clones are created unpublished so
  you can review them first.

#### Clone Workflow

1. Go to the **Role Variants** tab on your primary node.
2. In "Add role variant", select **Clone this node**.
3. Select the target **Role**.
4. Enter a **Variant title** (e.g., "Homepage (Anonymous)").
5. Choose whether to clone paragraphs.
6. Click **Add variant**.
7. You'll be shown a link to edit the new variant - click it to customize.
8. Publish the variant when ready.

### Shared Path vs Separate Paths

The module supports two modes of operation:

#### Shared Path (Default)

When enabled, all variants share the same URL path. Users visiting the primary
node's URL will see content appropriate for their role without any redirect.

- URL stays the same: `/node/1` or `/my-page`
- Content is swapped based on user role
- Better for SEO and user experience
- Ideal for personalized landing pages

#### Separate Paths

When disabled, each variant maintains its own URL. Users are redirected to their
appropriate variant.

- Each variant has its own path
- 302 redirects route users to correct variant
- Useful when variants need distinct URLs
- Good for tracking/analytics per role

To toggle this setting:
1. Go to the **Role Variants** tab on the primary node.
2. Expand **Settings**.
3. Check or uncheck **Share path alias across variants**.

### Priority Weights

When a user has multiple roles, the weight determines which variant takes
priority:

- Lower weight = Higher priority
- Drag and drop variants in the table to reorder
- The first matching role (by weight) determines the variant shown

**Example**: A user with both "subscriber" and "premium" roles will see the
variant with the lower weight value.

## How It Works

### Request Flow

1. User requests a node page (e.g., `/node/1`).
2. The `NodeViewSubscriber` intercepts the request at priority 28 (after routing
   but before page cache).
3. The module checks if role variants are enabled for this content type.
4. It looks up the variant set for the requested node.
5. Based on the user's roles and variant weights, it determines the appropriate
   variant.
6. **Shared path mode**: The node is swapped in the request, and the RouteMatch
   is reset so the variant renders as if it were the canonical page.
7. **Separate path mode**: A 302 redirect is issued to the variant's URL.

### Variant Resolution

The module resolves variants using this logic:

1. Get the variant set for the current node (works whether user lands on primary
   or any variant).
2. Sort variants by weight (ascending).
3. Iterate through variants and find the first one matching any of the user's
   roles.
4. If a match is found and it's different from the current node, swap or
   redirect.
5. If no match, show the primary node.

## Template Suggestions

The module adds template suggestions for both page and node templates:

### Page Templates

- `page--node-role-variant.html.twig`
- `page--node-role-variant--[variant-set-id].html.twig`
- `page--node--[bundle]--role-variant.html.twig`
- `page--node--[bundle]--role-variant--[variant-set-id].html.twig`

### Node Templates

- `node--role-variant.html.twig`
- `node--role-variant--[view-mode].html.twig`
- `node--role-variant--[variant-set-id].html.twig`
- `node--role-variant--[variant-set-id]--[view-mode].html.twig`
- `node--[bundle]--role-variant.html.twig`
- `node--[bundle]--role-variant--[view-mode].html.twig`
- `node--[bundle]--role-variant--[variant-set-id].html.twig`
- `node--[bundle]--role-variant--[variant-set-id]--[view-mode].html.twig`

## Caching

The module properly integrates with Drupal's cache system:

### Cache Contexts

- `user.roles`: Ensures pages are cached per role combination.
- `url.query_args:no-role-redirect`: Respects the admin preview bypass.

### Cache Tags

- `node:[nid]`: Tags for primary and variant nodes.
- `config:node_role_variants.variant_set.[id]`: Invalidates when variant
  configuration changes.

### Admin Preview

Add `?no-role-redirect=1` to any URL to bypass role-based redirects/swapping.
This is useful for administrators testing different variants.

## API Reference

### NodeRoleVariantsManager Service

The `node_role_variants.manager` service provides programmatic access. All methods
use node UUIDs (not node IDs) for cross-environment portability:

```php
/** @var \Drupal\node_role_variants\Service\NodeRoleVariantsManager $manager */
$manager = \Drupal::service('node_role_variants.manager');

// Get variant for a specific user.
$variant = $manager->getVariantForUser($node, $user);

// Check if node is a primary node (has variants).
$is_primary = $manager->isPrimaryNode($node->uuid());

// Check if node is a variant.
$is_variant = $manager->isVariantNode($node->uuid());

// Get the primary node UUID for a variant.
$primary_uuid = $manager->getPrimaryNodeUuid($variant_node->uuid());

// Get all variants for a primary node.
$variants = $manager->getVariants($node->uuid());

// Add a variant programmatically.
$manager->addVariant($primary_uuid, $variant_uuid, 'authenticated', $weight);

// Remove a variant.
$manager->removeVariant($primary_uuid, 'authenticated');

// Check if role variants are enabled for a node.
$enabled = $manager->isEnabledForNode($node);

// Get/set shared path setting.
$shared = $manager->getSharedPathForNode($node->uuid());
$manager->setSharedPath($primary_uuid, TRUE);

// Load a node by UUID.
$node = $manager->loadNodeByUuid($uuid);
```

### Hooks

#### hook_entity_delete()

The module automatically cleans up variant relationships when nodes are deleted.

#### hook_form_node_type_form_alter()

Adds the "Role variants" configuration to content type edit forms.

### Helper Function

```php
// Check if role variants are enabled for a content type.
$enabled = node_role_variants_is_enabled('article');
```

## Troubleshooting

### Variant Not Showing

1. **Clear cache**: Run `drush cr` after configuration changes.
2. **Check permissions**: Ensure the user has access to view the variant node.
3. **Verify role assignment**: Confirm the user has the expected role.
4. **Check weights**: If user has multiple roles, verify weight priorities.
5. **Use admin preview**: Add `?no-role-redirect=1` to see what variant would
   show.

### Title Displaying When It Shouldn't

If the variant node's title is displaying when it shouldn't:

1. Ensure you're using the latest version of the module.
2. Clear the Drupal cache completely.
3. The module resets the RouteMatch cache to ensure `node_is_page()` returns
   correctly for swapped nodes.

### Redirect Loops

If you experience redirect loops:

1. Ensure variant nodes are not also primary nodes of other variant sets.
2. Check that a node is not configured as its own variant.
3. Verify the variant set configuration is consistent.

### Caching Issues

For caching-related issues:

1. Ensure `user.roles` cache context is not being stripped.
2. Check that cache tags are properly invalidating.
3. Verify Dynamic Page Cache is running at the expected priority.
