<?php

namespace Drupal\node_role_variants\Entity;

use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Interface for Node Role Variant Set configuration entities.
 */
interface NodeRoleVariantSetInterface extends ConfigEntityInterface {

  /**
   * Gets the primary node UUID.
   *
   * @return string
   *   The primary node UUID.
   */
  public function getPrimaryUuid();

  /**
   * Sets the primary node UUID.
   *
   * @param string $uuid
   *   The primary node UUID.
   *
   * @return $this
   */
  public function setPrimaryUuid($uuid);

  /**
   * Gets whether to share path alias across variants.
   *
   * @return bool
   *   TRUE if shared path is enabled.
   */
  public function getSharedPath();

  /**
   * Sets whether to share path alias across variants.
   *
   * @param bool $shared_path
   *   TRUE to share path alias.
   *
   * @return $this
   */
  public function setSharedPath($shared_path);

  /**
   * Gets all role variants.
   *
   * @return array
   *   Array of variants keyed by role_id.
   */
  public function getVariants();

  /**
   * Sets all role variants.
   *
   * @param array $variants
   *   Array of variants.
   *
   * @return $this
   */
  public function setVariants(array $variants);

  /**
   * Adds a variant for a role.
   *
   * @param string $role_id
   *   The role machine name.
   * @param string $variant_uuid
   *   The variant node UUID.
   * @param int $weight
   *   The weight for priority ordering.
   *
   * @return $this
   */
  public function addVariant($role_id, $variant_uuid, $weight = 0);

  /**
   * Removes a variant for a role.
   *
   * @param string $role_id
   *   The role machine name.
   *
   * @return $this
   */
  public function removeVariant($role_id);

  /**
   * Gets a specific variant by role.
   *
   * @param string $role_id
   *   The role machine name.
   *
   * @return array|null
   *   The variant data or NULL.
   */
  public function getVariant($role_id);

  /**
   * Checks if a variant exists for a role.
   *
   * @param string $role_id
   *   The role machine name.
   *
   * @return bool
   *   TRUE if variant exists.
   */
  public function hasVariant($role_id);

  /**
   * Gets the variant node UUID for a specific role.
   *
   * @param string $role_id
   *   The role machine name.
   *
   * @return string|null
   *   The variant node UUID or NULL.
   */
  public function getVariantUuidForRole($role_id);

}
