<?php

namespace Drupal\node_role_variants\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Drupal\node_role_variants\Service\NodeRoleVariantsManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for managing role variants of a node.
 */
class NodeRoleVariantsForm extends FormBase {

  /**
   * The role variants manager.
   *
   * @var \Drupal\node_role_variants\Service\NodeRoleVariantsManager
   */
  protected $variantsManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a NodeRoleVariantsForm object.
   *
   * @param \Drupal\node_role_variants\Service\NodeRoleVariantsManager $variants_manager
   *   The role variants manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    NodeRoleVariantsManager $variants_manager,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    $this->variantsManager = $variants_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('node_role_variants.manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_role_variants_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {
    if (!$node) {
      return $form;
    }

    $form_state->set('node', $node);
    $uuid = $node->uuid();

    // Check if this node is a variant of another node.
    $variant_info = $this->variantsManager->getVariantInfo($uuid);
    if ($variant_info) {
      $primary_node = $this->variantsManager->loadNodeByUuid($variant_info['primary_uuid']);
      $role = $this->entityTypeManager->getStorage('user_role')->load($variant_info['role_id']);

      $form['variant_notice'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        'message' => [
          '#markup' => $this->t('This node is a role variant of @link for the "@role" role. To manage role variants, go to the primary node.', [
            '@link' => $primary_node ? Link::fromTextAndUrl($primary_node->label(), Url::fromRoute('node_role_variants.node_variants', ['node' => $primary_node->id()]))->toString() : $this->t('Unknown node'),
            '@role' => $role ? $role->label() : $variant_info['role_id'],
          ]),
        ],
      ];

      return $form;
    }

    // Get all roles.
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $role_options = [];
    foreach ($roles as $role) {
      $role_options[$role->id()] = $role->label();
    }

    // Get existing variants.
    $variants = $this->variantsManager->getVariants($uuid);
    $used_roles = array_keys($variants);

    // Build the variants table.
    $form['variants'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Role'),
        $this->t('Content'),
        $this->t('Status'),
        $this->t('Weight'),
        $this->t('Operations'),
      ],
      '#empty' => $this->t('No role variants configured.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'variant-weight',
        ],
      ],
    ];

    // Add current node as default/fallback (not draggable).
    $form['variants']['_default'] = [
      '#attributes' => ['class' => ['draggable']],
      '#weight' => -1000,
      'role' => [
        '#markup' => '<strong>' . $this->t('Default (fallback)') . '</strong>',
      ],
      'content' => [
        '#markup' => Link::fromTextAndUrl($node->label(), $node->toUrl())->toString(),
      ],
      'status' => [
        '#markup' => $node->isPublished() ? $this->t('Published') : $this->t('Unpublished'),
      ],
      'weight' => [
        '#markup' => '-',
      ],
      'operations' => [
        '#markup' => '-',
      ],
    ];

    // Add existing variants.
    foreach ($variants as $role_id => $variant_data) {
      $variant_node = $this->variantsManager->loadNodeByUuid($variant_data['variant_uuid']);
      $role = $roles[$role_id] ?? NULL;

      $form['variants'][$role_id] = [
        '#attributes' => ['class' => ['draggable']],
        '#weight' => $variant_data['weight'],
        'role' => [
          '#markup' => $role ? $role->label() : $role_id,
        ],
        'content' => [
          '#markup' => $variant_node ? Link::fromTextAndUrl($variant_node->label(), $variant_node->toUrl())->toString() : $this->t('Node not found'),
        ],
        'status' => [
          '#markup' => $variant_node ? ($variant_node->isPublished() ? $this->t('Published') : $this->t('Unpublished')) : '-',
        ],
        'weight' => [
          '#type' => 'weight',
          '#title' => $this->t('Weight for @role', ['@role' => $role ? $role->label() : $role_id]),
          '#title_display' => 'invisible',
          '#default_value' => $variant_data['weight'],
          '#attributes' => ['class' => ['variant-weight']],
        ],
        'operations' => [
          '#type' => 'operations',
          '#links' => [
            'edit' => [
              'title' => $this->t('Edit'),
              'url' => $variant_node ? $variant_node->toUrl('edit-form') : Url::fromRoute('<none>'),
            ],
            'remove' => [
              'title' => $this->t('Remove'),
              'url' => Url::fromRoute('node_role_variants.remove_variant', [
                'node' => $node->id(),
                'role_id' => $role_id,
              ]),
            ],
          ],
        ],
      ];
    }

    // Add new variant section.
    $available_roles = array_diff_key($role_options, array_flip($used_roles));
    $is_new_variant_set = empty($variants) && !$this->variantsManager->isPrimaryNode($uuid);

    if (!empty($available_roles)) {
      $form['add_variant'] = [
        '#type' => 'details',
        '#title' => $this->t('Add role variant'),
        '#open' => TRUE,
      ];

      $form['add_variant']['method'] = [
        '#type' => 'radios',
        '#title' => $this->t('Method'),
        '#options' => [
          'reference' => $this->t('Reference existing content'),
          'clone' => $this->t('Clone this node'),
        ],
        '#default_value' => 'reference',
        '#description' => $this->t('<strong>Reference:</strong> Use an existing node as the variant.<br><strong>Clone:</strong> Create a copy of this node to use as the variant.'),
      ];

      // Show variant set ID/label fields when creating the first variant.
      if ($is_new_variant_set) {
        $form['add_variant']['variant_set_label'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Variant set label'),
          '#description' => $this->t('A human-readable label for this variant set.'),
          '#default_value' => $node->label(),
          '#required' => FALSE,
        ];

        $form['add_variant']['variant_set_id'] = [
          '#type' => 'machine_name',
          '#title' => $this->t('Variant set ID'),
          '#description' => $this->t('A unique machine name for this variant set. Used for template suggestions like <code>page--node-role-variant--[id].html.twig</code>.'),
          '#default_value' => '',
          '#machine_name' => [
            'exists' => [$this, 'variantSetIdExists'],
            'source' => ['add_variant', 'variant_set_label'],
            'replace_pattern' => '[^a-z0-9_]+',
            'replace' => '_',
          ],
          '#required' => FALSE,
        ];
      }

      $form['add_variant']['role'] = [
        '#type' => 'select',
        '#title' => $this->t('Role'),
        '#options' => $available_roles,
        '#empty_option' => $this->t('- Select a role -'),
        '#required' => FALSE,
      ];

      // Reference method: select existing node.
      $form['add_variant']['variant_node'] = [
        '#type' => 'entity_autocomplete',
        '#title' => $this->t('Content'),
        '#description' => $this->t('Select the node to display for users with this role.'),
        '#target_type' => 'node',
        '#selection_settings' => [
          'target_bundles' => [$node->bundle()],
        ],
        '#states' => [
          'visible' => [
            ':input[name="method"]' => ['value' => 'reference'],
          ],
        ],
      ];

      // Clone method: variant title.
      $form['add_variant']['clone_title'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Variant title'),
        '#description' => $this->t('The title for the cloned variant node.'),
        '#default_value' => $node->label() . ' (' . $this->t('Variant') . ')',
        '#maxlength' => 255,
        '#states' => [
          'visible' => [
            ':input[name="method"]' => ['value' => 'clone'],
          ],
        ],
      ];

      // Clone method: clone paragraphs option.
      $form['add_variant']['clone_paragraphs'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Clone paragraph references'),
        '#description' => $this->t('If enabled, paragraph entities will be duplicated. If disabled, the clone will share paragraphs with the original.'),
        '#default_value' => TRUE,
        '#states' => [
          'visible' => [
            ':input[name="method"]' => ['value' => 'clone'],
          ],
        ],
      ];

      // Clone method: publish option.
      $form['add_variant']['clone_published'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Publish the cloned variant'),
        '#description' => $this->t('If disabled, the clone will be created as unpublished.'),
        '#default_value' => FALSE,
        '#states' => [
          'visible' => [
            ':input[name="method"]' => ['value' => 'clone'],
          ],
        ],
      ];

      $form['add_variant']['add'] = [
        '#type' => 'submit',
        '#value' => $this->t('Add variant'),
        '#submit' => ['::addVariantSubmit'],
        '#validate' => ['::addVariantValidate'],
      ];
    }

    // Settings section (only show if there are variants or this is a primary).
    if (!empty($variants) || $this->variantsManager->isPrimaryNode($uuid)) {
      $variant_set = $this->variantsManager->getVariantSetByPrimaryUuid($uuid);

      $form['settings'] = [
        '#type' => 'details',
        '#title' => $this->t('Settings'),
        '#open' => TRUE,
      ];

      $form['settings']['variant_set_label'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Variant set label'),
        '#description' => $this->t('A human-readable label for this variant set.'),
        '#default_value' => $variant_set ? $variant_set->label() : $node->label(),
        '#required' => TRUE,
      ];

      $form['settings']['variant_set_id'] = [
        '#type' => 'machine_name',
        '#title' => $this->t('Variant set ID'),
        '#description' => $this->t('A unique machine name for this variant set. Used for template suggestions like <code>page--node-role-variant--[id].html.twig</code>.'),
        '#default_value' => $variant_set ? $variant_set->id() : 'node_' . substr($uuid, 0, 8),
        '#machine_name' => [
          'exists' => [$this, 'variantSetIdExists'],
          'source' => ['settings', 'variant_set_label'],
          'replace_pattern' => '[^a-z0-9_]+',
          'replace' => '_',
        ],
        '#disabled' => $variant_set !== NULL,
        '#required' => TRUE,
      ];

      if ($variant_set) {
        $form['settings']['variant_set_id']['#description'] = $this->t('The variant set ID cannot be changed after creation. Current ID: <code>@id</code>', [
          '@id' => $variant_set->id(),
        ]);
      }

      $form['settings']['shared_path'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Share path alias across variants'),
        '#description' => $this->t('When enabled, all role variants will use the same URL path as the primary node. When disabled, each variant can have its own unique path alias.'),
        '#default_value' => $this->variantsManager->getSharedPath($uuid),
      ];
    }

    // Actions section.
    $form['actions'] = [
      '#type' => 'actions',
    ];

    if (!empty($variants)) {
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
      ];
    }

    return $form;
  }

  /**
   * Validation handler for adding a variant.
   */
  public function addVariantValidate(array &$form, FormStateInterface $form_state) {
    $role = $form_state->getValue('role');
    $method = $form_state->getValue('method');

    if (empty($role)) {
      $form_state->setErrorByName('role', $this->t('Please select a role.'));
      return;
    }

    if ($method === 'reference') {
      // Reference method validation.
      $variant_node_id = $form_state->getValue('variant_node');

      if (empty($variant_node_id)) {
        $form_state->setErrorByName('variant_node', $this->t('Please select a node.'));
        return;
      }

      /** @var \Drupal\node\NodeInterface $node */
      $node = $form_state->get('node');
      $variant_node = $this->entityTypeManager->getStorage('node')->load($variant_node_id);

      if (!$variant_node) {
        $form_state->setErrorByName('variant_node', $this->t('The selected node could not be loaded.'));
        return;
      }

      // Check for same node.
      if ($node->uuid() === $variant_node->uuid()) {
        $form_state->setErrorByName('variant_node', $this->t('A node cannot be its own variant.'));
        return;
      }

      // Check if the variant node is already a primary or variant.
      $variant_uuid = $variant_node->uuid();
      if ($this->variantsManager->isPrimaryNode($variant_uuid)) {
        $form_state->setErrorByName('variant_node', $this->t('This node is already a primary node with variants. It cannot be used as a variant.'));
      }
      elseif ($this->variantsManager->isVariantNode($variant_uuid)) {
        $form_state->setErrorByName('variant_node', $this->t('This node is already a variant of another node.'));
      }
    }
    elseif ($method === 'clone') {
      // Clone method validation.
      $clone_title = $form_state->getValue('clone_title');

      if (empty($clone_title)) {
        $form_state->setErrorByName('clone_title', $this->t('Please provide a title for the cloned variant.'));
      }
    }
  }

  /**
   * Submit handler for adding a variant.
   */
  public function addVariantSubmit(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\node\NodeInterface $node */
    $node = $form_state->get('node');
    $uuid = $node->uuid();
    $role = $form_state->getValue('role');
    $method = $form_state->getValue('method');

    // Get custom variant set ID and label if provided (for new variant sets).
    $variant_set_id = $form_state->getValue('variant_set_id') ?: NULL;
    $variant_set_label = $form_state->getValue('variant_set_label') ?: $node->label();

    // Get the next weight.
    $variants = $this->variantsManager->getVariants($uuid);
    $max_weight = 0;
    foreach ($variants as $variant) {
      if ($variant['weight'] > $max_weight) {
        $max_weight = $variant['weight'];
      }
    }

    try {
      if ($method === 'clone') {
        // Clone the node and use it as the variant.
        $clone_title = $form_state->getValue('clone_title');
        $clone_paragraphs = (bool) $form_state->getValue('clone_paragraphs');
        $clone_published = (bool) $form_state->getValue('clone_published');

        $clone = $this->cloneNode($node, $clone_title, $clone_paragraphs, $clone_published);

        if (!$clone) {
          $this->messenger()->addError($this->t('Failed to create the cloned node.'));
          $form_state->setRebuild();
          return;
        }

        $variant_uuid = $clone->uuid();

        $this->variantsManager->addVariant($uuid, $variant_uuid, $role, $max_weight + 1, $variant_set_id, $variant_set_label);

        $role_entity = $this->entityTypeManager->getStorage('user_role')->load($role);
        $role_label = $role_entity ? $role_entity->label() : $role;

        $this->messenger()->addStatus($this->t('Created cloned variant "@title" for the "@role" role. <a href="@edit_url">Edit the variant</a>.', [
          '@title' => $clone->label(),
          '@role' => $role_label,
          '@edit_url' => $clone->toUrl('edit-form')->toString(),
        ]));
      }
      else {
        // Reference method: use existing node.
        $variant_node_id = $form_state->getValue('variant_node');
        $variant_node = $this->entityTypeManager->getStorage('node')->load($variant_node_id);
        $variant_uuid = $variant_node->uuid();

        $this->variantsManager->addVariant($uuid, $variant_uuid, $role, $max_weight + 1, $variant_set_id, $variant_set_label);
        $this->messenger()->addStatus($this->t('Role variant added successfully.'));
      }
    }
    catch (\InvalidArgumentException $e) {
      $this->messenger()->addError($e->getMessage());
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('An error occurred: @message', [
        '@message' => $e->getMessage(),
      ]));
    }

    $form_state->setRebuild();
  }

  /**
   * Clones a node entity.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to clone.
   * @param string $title
   *   The title for the cloned node.
   * @param bool $clone_paragraphs
   *   Whether to clone paragraph references.
   * @param bool $published
   *   Whether the clone should be published.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The cloned node, or NULL on failure.
   */
  protected function cloneNode(NodeInterface $node, string $title, bool $clone_paragraphs, bool $published): ?NodeInterface {
    /** @var \Drupal\node\NodeInterface $clone */
    $clone = $node->createDuplicate();

    // Set the new title.
    $clone->setTitle($title);

    // Set published status.
    if ($published) {
      $clone->setPublished();
    }
    else {
      $clone->setUnpublished();
    }

    // Reset some fields that shouldn't be cloned.
    $clone->setCreatedTime(\Drupal::time()->getRequestTime());
    $clone->setChangedTime(\Drupal::time()->getRequestTime());
    $clone->setRevisionCreationTime(\Drupal::time()->getRequestTime());
    $clone->setOwnerId(\Drupal::currentUser()->id());
    $clone->setRevisionUserId(\Drupal::currentUser()->id());

    // Handle paragraph fields if requested.
    if ($clone_paragraphs) {
      $this->cloneParagraphFields($clone);
    }

    // Save the clone.
    $clone->save();

    return $clone;
  }

  /**
   * Clones paragraph field references on a node.
   *
   * @param \Drupal\node\NodeInterface $clone
   *   The cloned node (not yet saved).
   */
  protected function cloneParagraphFields(NodeInterface $clone): void {
    // Check if the Paragraphs module is enabled.
    if (!\Drupal::moduleHandler()->moduleExists('paragraphs')) {
      return;
    }

    $field_definitions = $clone->getFieldDefinitions();

    foreach ($field_definitions as $field_name => $field_definition) {
      $field_type = $field_definition->getType();

      // Check for entity_reference_revisions fields (used by Paragraphs).
      if ($field_type === 'entity_reference_revisions') {
        $settings = $field_definition->getSettings();
        $target_type = $settings['target_type'] ?? '';

        if ($target_type === 'paragraph') {
          $this->cloneParagraphField($clone, $field_name);
        }
      }
    }
  }

  /**
   * Clones a single paragraph field.
   *
   * @param \Drupal\node\NodeInterface $clone
   *   The cloned node.
   * @param string $field_name
   *   The field name.
   */
  protected function cloneParagraphField(NodeInterface $clone, string $field_name): void {
    if (!$clone->hasField($field_name)) {
      return;
    }

    $field = $clone->get($field_name);
    $new_paragraphs = [];

    foreach ($field as $item) {
      $paragraph = $item->entity;
      if ($paragraph) {
        $cloned_paragraph = $this->cloneParagraphRecursive($paragraph);
        if ($cloned_paragraph) {
          $new_paragraphs[] = $cloned_paragraph;
        }
      }
    }

    $clone->set($field_name, $new_paragraphs);
  }

  /**
   * Recursively clones a paragraph and its nested paragraphs.
   *
   * @param object $paragraph
   *   The paragraph to clone.
   *
   * @return object|null
   *   The cloned paragraph.
   */
  protected function cloneParagraphRecursive($paragraph) {
    if (!$paragraph) {
      return NULL;
    }

    $clone = $paragraph->createDuplicate();

    // Process nested paragraph fields.
    $field_definitions = $clone->getFieldDefinitions();
    foreach ($field_definitions as $field_name => $field_definition) {
      if ($field_definition->getType() === 'entity_reference_revisions') {
        $settings = $field_definition->getSettings();
        if (($settings['target_type'] ?? '') === 'paragraph') {
          $nested_paragraphs = [];
          foreach ($clone->get($field_name) as $item) {
            if ($item->entity) {
              $nested_clone = $this->cloneParagraphRecursive($item->entity);
              if ($nested_clone) {
                $nested_paragraphs[] = $nested_clone;
              }
            }
          }
          $clone->set($field_name, $nested_paragraphs);
        }
      }
    }

    return $clone;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\node\NodeInterface $node */
    $node = $form_state->get('node');
    $uuid = $node->uuid();

    // Update weights.
    $variants = $form_state->getValue('variants');
    if ($variants) {
      $weights = [];
      foreach ($variants as $role_id => $values) {
        if ($role_id !== '_default' && isset($values['weight'])) {
          $weights[$role_id] = $values['weight'];
        }
      }

      if (!empty($weights)) {
        $this->variantsManager->updateWeights($uuid, $weights);
      }
    }

    // Save variant set label.
    $variant_set_label = $form_state->getValue('variant_set_label');
    if ($variant_set_label) {
      $variant_set = $this->variantsManager->getVariantSetByPrimaryUuid($uuid);
      if ($variant_set) {
        $variant_set->set('label', $variant_set_label);
        $variant_set->save();
      }
    }

    // Save shared_path setting.
    $shared_path = $form_state->getValue('shared_path');
    if ($shared_path !== NULL) {
      $this->variantsManager->setSharedPath($uuid, (bool) $shared_path);
    }

    $this->messenger()->addStatus($this->t('Settings saved.'));
  }

  /**
   * Checks if a variant set ID already exists.
   *
   * @param string $id
   *   The variant set ID to check.
   *
   * @return bool
   *   TRUE if the ID exists, FALSE otherwise.
   */
  public function variantSetIdExists($id) {
    $variant_set = $this->entityTypeManager->getStorage('node_role_variant_set')->load($id);
    return $variant_set !== NULL;
  }

}
