<?php

namespace Drupal\nodehive_entity_access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeAccessControlHandler;

/**
 * Access controller for entities with space-based access control.
 *
 * This class provides an EntityAccessControlHandler implementation that
 * controls access to entities based on space membership.
 */
class NodeHiveEntityAccessControlHandler extends NodeAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    // Always allow users with 'bypass node access' or admin permission full access.
    if ($account->hasPermission('bypass node access') || $account->hasPermission('administer nodehive')) {
      return AccessResult::allowed()->setCacheMaxAge(0);
    }

    // Get the NodeHive entity access manager service.
    $nodehive_access_manager = \Drupal::service('nodehive_entity_access.manager');

    // For view operations, we need to be especially strict.
    if ($operation === 'view') {
      // Get spaces that contain this node.
      $space_storage = \Drupal::entityTypeManager()->getStorage('nodehive_space');
      $space_query = $space_storage->getQuery()
        ->condition('content', $entity->id(), 'IN')
        ->accessCheck(FALSE);
      $space_ids = $space_query->execute();

      // If this node isn't in any space, deny access.
      if (empty($space_ids)) {
        return AccessResult::forbidden("Entity is not in any space")->setCacheMaxAge(0);
      }

      // Find spaces that this user is a member of.
      $user_space_query = $space_storage->getQuery()
        ->condition('id', array_values($space_ids), 'IN')
        ->condition('editors', $account->id(), 'IN')
        ->accessCheck(FALSE);
      $user_space_ids = $user_space_query->execute();

      // If the user is not a member of any space that contains this node, deny access.
      if (empty($user_space_ids)) {
        return AccessResult::forbidden("User is not a member of any space containing this entity")->setCacheMaxAge(0);
      }

      // Allow access if the user is in a space with this entity.
      return AccessResult::allowed()->setCacheMaxAge(0);
    }

    // For other operations, use the manager service.
    $has_access = $nodehive_access_manager->userHasEntityAccess($entity->id(), $entity->getEntityTypeId(), $account, $operation);

    return $has_access ?
      AccessResult::allowed()->setCacheMaxAge(0) :
      AccessResult::forbidden()->setCacheMaxAge(0);
  }

  /**
   * {@inheritdoc}
   */
  public function access(EntityInterface $entity, $operation, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    $account = $this->prepareUser($account);

    // For view operations, always use our strict checkAccess implementation.
    if ($operation === 'view') {
      $result = $this->checkAccess($entity, $operation, $account);
      return $return_as_object ? $result : $result->isAllowed();
    }

    // For other operations, use the parent implementation.
    return parent::access($entity, $operation, $account, $return_as_object);
  }

}
