<?php

namespace Drupal\nodehive_public_api\Controller;

use Drupal\Core\Cache\CacheableJsonResponse;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for API index endpoint.
 */
class ApiController extends ControllerBase {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs an ApiController object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    LanguageManagerInterface $language_manager,
    RequestStack $request_stack,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    $this->languageManager = $language_manager;
    $this->requestStack = $request_stack;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('request_stack'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Redirects from /nodehive-api to /nodehive-api/v1.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   The redirect response.
   */
  public function redirectToV1(): RedirectResponse {
    $current_language = $this->languageManager->getCurrentLanguage();
    $language_prefixes = \Drupal::config('language.negotiation')->get('url.prefixes');
    $prefix = !empty($language_prefixes[$current_language->getId()]) ? '/' . $language_prefixes[$current_language->getId()] : '';

    $url = $prefix . '/nodehive-api/v1';
    return new RedirectResponse($url, 301);
  }

  /**
   * Returns the list of available API endpoints.
   *
   * @return \Drupal\Core\Cache\CacheableJsonResponse
   *   The JSON response containing available endpoints.
   */
  public function index(): CacheableJsonResponse {
    try {
      $request = $this->requestStack->getCurrentRequest();
      $base_url = $request->getSchemeAndHttpHost();
      $languages = $this->languageManager->getLanguages();
      $language_codes = array_keys($languages);

      // Get language prefix configuration.
      $language_prefixes = \Drupal::config('language.negotiation')->get('url.prefixes');

      // System menus that should not be exposed via API.
      $excluded_menus = [
        'account',
        'admin',
        'devel',
        'nodehive-admin',
        'nodehive-crm',
        'nodehive-editor',
        'tools',
      ];

      // Load all available menus.
      $menu_storage = $this->entityTypeManager->getStorage('menu');
      $menus = $menu_storage->loadMultiple();

    $available_menus = [];
    foreach ($menus as $menu) {
      $menu_id = $menu->id();

      // Skip system menus.
      if (in_array($menu_id, $excluded_menus)) {
        continue;
      }

      $menu_urls = [];

      // Build URLs for each language.
      foreach ($language_codes as $langcode) {
        $prefix = !empty($language_prefixes[$langcode]) ? '/' . $language_prefixes[$langcode] : '';
        $menu_urls[$langcode] = $base_url . $prefix . '/nodehive-api/v1/menu/' . $menu_id;
      }

      $available_menus[] = [
        'id' => $menu_id,
        'label' => $menu->label(),
        'description' => $menu->getDescription(),
        'urls' => $menu_urls,
      ];
    }

    // Load all available spaces.
    $space_storage = $this->entityTypeManager->getStorage('nodehive_space');
    $spaces = $space_storage->loadMultiple();

    $available_spaces = [];
    $available_spaces_with_urls = [];
    foreach ($spaces as $space) {
      $space_id = $space->id();
      $space_urls = [];

      // Build URLs for each language.
      foreach ($language_codes as $langcode) {
        $prefix = !empty($language_prefixes[$langcode]) ? '/' . $language_prefixes[$langcode] : '';
        $space_urls[$langcode] = $base_url . $prefix . '/nodehive-api/v1/space/' . $space_id;
      }

      $available_spaces[] = [
        'id' => $space_id,
        'name' => $space->label(),
      ];

      $available_spaces_with_urls[] = [
        'id' => $space_id,
        'name' => $space->label(),
        'urls' => $space_urls,
      ];
    }

    // Build endpoint documentation.
    $first_menu = reset($menus);
    $first_menu_id = $first_menu ? $first_menu->id() : 'main';
    $example_prefix = !empty($language_prefixes[$language_codes[0]]) ? '/' . $language_prefixes[$language_codes[0]] : '';

    $endpoints = [
      [
        'name' => 'Menu API',
        'method' => 'GET',
        'available_menus' => $available_menus,
      ],
      [
        'name' => 'Spaces API',
        'method' => 'GET',
        'available_spaces' => $available_spaces_with_urls,
      ],
    ];

    // Build response.
    $response_data = [
      'name' => 'NodeHive Public API',
      'version' => '1.0',
      'info' => [
        'available_languages' => $language_codes,
        'available_spaces' => $available_spaces,
      ],
      'endpoints' => $endpoints,
    ];

    $response = new CacheableJsonResponse($response_data, 200, [], FALSE);

    // Add cache metadata.
    $cache_metadata = new CacheableMetadata();
    $cache_metadata->addCacheContexts(['user.permissions', 'languages']);
    $cache_metadata->addCacheTags(['config:core.extension', 'config:system.menu_list', 'nodehive_space_list']);
    $response->addCacheableDependency($cache_metadata);

    // Add cache tags for all menus.
    foreach ($menus as $menu) {
      $response->addCacheableDependency($menu);
    }

      // Add cache tags for all spaces.
      foreach ($spaces as $space) {
        $response->addCacheableDependency($space);
      }

      return $response;
    }
    catch (\Exception $e) {
      \Drupal::logger('nodehive_public_api')->error(
        'Error generating API index: @message',
        ['@message' => $e->getMessage()]
      );

      // Return minimal error response.
      $error_response = new CacheableJsonResponse([
        'error' => $this->t('An error occurred while generating the API index.'),
      ], 500);

      $cache_metadata = new CacheableMetadata();
      $cache_metadata->addCacheContexts(['user.permissions']);
      $error_response->addCacheableDependency($cache_metadata);

      return $error_response;
    }
  }

}
