<?php

namespace Drupal\nodehive_core\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\node\NodeInterface;
use Drupal\nodehive_core\Helper\NodehiveLanguageHelper;

/**
 * Controller for rendering node preview in the 'rag' view mode.
 */
class NodehiveActivityBlockController extends ControllerBase
{

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The language helper service.
   *
   * @var \Drupal\nodehive_core\Helper\NodehiveLanguageHelper
   */
  protected $languageHelper;

  /**
   * Constructs a NodehiveActivityBlockController.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\nodehive_core\Helper\NodehiveLanguageHelper $language_helper
   *   The language helper service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    DateFormatterInterface $date_formatter,
    AccountProxyInterface $current_user,
    RendererInterface $renderer,
    NodehiveLanguageHelper $language_helper
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->dateFormatter = $date_formatter;
    $this->currentUser = $current_user;
    $this->renderer = $renderer;
    $this->languageHelper = $language_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('date.formatter'),
      $container->get('current_user'),
      $container->get('renderer'),
      $container->get('nodehive_core.language_helper')
    );
  }

  /**
   * Renders a preview of nodes based on the action.
   *
   * @param string $action
   *   The action to determine which nodes to render.
   * @param string $number_of_items
   *   The number of items.
   * @param string $workflow
   *   The workflow.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response containing the rendered content.
   */
  public function renderAction($action, $number_of_items, $workflow)
  {
    $nodes = ($action === 'assigned-to-me') ? $this->getAssignedToMeContent($number_of_items) : $this->getTeamContent($number_of_items);

    if (empty($nodes)) {
      $build = [
        '#type' => 'markup',
        '#markup' => '<div class="no-activity">' . $this->t('No activity') . '</div>',
      ];
      return new Response($this->renderer->renderRoot($build));
    }

    $langcodes = array_keys($this->languageManager()->getLanguages());
    $langcode_header = strtoupper(implode(" ", $langcodes));

    $header = [
      'title' => $this->t('Title'),
      'content_type' => $this->t('Content Type'),
      'space' => "Space",
      'language' => $this->t('Languages'),

      //'workflow_state' => $this->t('Workflow State'),
      'author' => $this->t('Author'),
      'status' => $this->t('Status'),
      'last_edited' => $this->t('Last update'),
    ];

    $rows = $this->buildNodeRows($nodes, $workflow);
    $build = [
      '#type' => 'table',
      '#header' => $header,
    ];

    foreach ($rows as $row) {
      $build[] = $row;
    }

    $build['#attached']['library'][] = 'nodehive_core/activity-block';

    return new Response($this->renderer->renderRoot($build));
  }

  /**
   * Builds the rows for the nodes table.
   *
   * @param array $nodes
   *   The nodes to render.
   * @param string $workflow
   *   The workflow.
   *
   * @return array
   *   An array of table rows.
   */
  protected function buildNodeRows(array $nodes, $workflow)
  {
    $rows = [];
    foreach ($nodes as $node) {
      assert($node instanceof NodeInterface);
      $node_type = $this->entityTypeManager->getStorage('node_type')->load($node->bundle());
      $rows[] = [
        'title' => [
          '#type' => 'link',
          '#title' => $node->label(),
          '#url' => Url::fromRoute('entity.node.canonical', ['node' => $node->id()]),
          '#attributes' => [
            'class' => [
              'truncate',
              'text-ellipsis',
              'block',
              'max-w-72',
            ],
          ],
        ],
        'content_type' => [
          '#type' => 'markup',
          '#markup' => '<span class="truncate text-ellipsis block max-w-36">' . ($node_type ? $node_type->label() : $node->bundle()) . '</span>',
        ],
        'space' => ['#markup' => $this->getSpaceLink($node)],
        'language' => $this->buildLanguageSwitcher($node),

        //'workflow_state' => ['#markup' => $this->getWorkflowState($node, $workflow)],
        'author' => ['#markup' => $this->getAuthorName($node)],
        'status' => ['#markup' => $this->buildNodeStatus($node)],
        'last_edited' => [
          '#type' => 'markup',
          '#markup' => '<span class="truncate text-ellipsis block max-w-36 text-sm text-gray-600">' . $this->dateFormatter->formatInterval(REQUEST_TIME - $node->getChangedTime(), 1) . '</span>',
        ],

        //'edit_link' => ['#markup' => $this->buildLink($node, 'edit')],
      ];
    }
    return $rows;
  }

  /**
   * Builds a link for a node.
   */
  protected function buildNodeLink($node)
  {
    $node_url = Url::fromRoute('entity.node.canonical', ['node' => $node->id()]);
    $node_link = Link::fromTextAndUrl($node->label(), $node_url)->toString();
    return $node_link;
  }

  /**
   * Builds a status for a node.
   */
  protected function buildNodeStatus($node)
  {
    $status_class = $node->isPublished() ? "published" : "unpublished";
    $status = $node->isPublished() ? $this->t('Published') : $this->t('Unpublished');

    $markup = "<span class='marker marker--" . strtolower($status_class) . "'>$status</span>";
    return $markup;
  }

  /**
   * Builds a language switcher for a node.
   */
  protected function buildLanguageSwitcher($entity)
  {
    return $this->languageHelper->buildLanguageEditLinks($entity);
  }

  /**
   * Builds a link for a node.
   */
  protected function buildLink($node, $type = 'view')
  {
    $route = ($type === 'edit') ? 'entity.node.edit_form' : 'entity.node.canonical';

    $link = [
      '#type' => 'link',
      '#url' => Url::fromRoute($route, ['node' => $node->id()]),
      '#title' => $this->t(ucfirst($type)),
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
    ];

    $link = $this->renderer->renderRoot($link);

    return $link;
  }

  /**
   * Gets the workflow state for a node.
   */
  protected function getWorkflowState($node, $workflow)
  {
    $state = '';
    if ($node->hasField('moderation_state') && !$node->get('moderation_state')->isEmpty()) {
      $workflow = $this->entityTypeManager->getStorage('workflow')->load($workflow);
      $state_plugin = $workflow ? $workflow->getTypePlugin()->getState($node->get('moderation_state')->value) : NULL;
      $state = $state_plugin ? $state_plugin->label() : '';
    }

    $markup = "<span class='marker marker--" . strtolower($state) . "'>$state</span>";
    return $markup;
  }

  /**
   * Gets the author's name for a node.
   */
  protected function getAuthorName(NodeInterface $node)
  {
    $author_name = '';
    $max_length = 14;

    if ($node->hasField('uid') && !$node->get('uid')->isEmpty()) {
      $author = $this->entityTypeManager->getStorage('user')->load($node->get('uid')->target_id);
      if ($author) {
        $full_name = $author->getDisplayName();
        if (mb_strlen($full_name) > $max_length) {
          $author_name = mb_substr($full_name, 0, $max_length) . '...';
        } else {
          $author_name = $full_name;
        }
      }
    }

    $link = [
      '#type' => 'link',
      '#url' => $node->getOwner()->toUrl(),
      '#title' => $author_name,
      '#attributes' => [
        'class' => ['username'],
      ],
    ];

    $link = $this->renderer->renderRoot($link);

    return $link;
  }

  /**
   * Gets the space link for a node.
   */
  protected function getSpaceLink($node)
  {
    if ($node->hasField('nodehive_space') && !$node->get('nodehive_space')->isEmpty()) {
      $space = $this->entityTypeManager->getStorage('nodehive_space')->load($node->get('nodehive_space')->target_id);
      if ($space) {
        $link = Link::fromTextAndUrl($space->label(), Url::fromRoute('entity.nodehive_space.canonical', ['nodehive_space' => $space->id()]));
        $link_render = $link->toRenderable();
        $link_render['#attributes']['class'] = ['truncate', 'text-ellipsis', 'block', 'max-w-36'];
        return $this->renderer->renderRoot($link_render);
      }
      return $this->t('not set');
    }
    return $this->t('not set');
  }

  /**
   * Retrieves nodes assigned to the current user.
   */
  protected function getAssignedToMeContent($number_of_items)
  {
    return $this->loadNodes($number_of_items, ['uid' => $this->currentUser->id()]);
  }

  /**
   * Retrieves team nodes.
   */
  protected function getTeamContent($number_of_items)
  {
    return $this->loadNodes($number_of_items);
  }

  /**
   * Loads nodes with optional conditions.
   */
  protected function loadNodes($number_of_items, array $conditions = [])
  {
    $query = $this->entityTypeManager->getStorage('node')->getQuery()->accessCheck(TRUE)->sort('changed', 'DESC')->range(0, $number_of_items);

    // Get all spaces that the current user has access.
    $spaces_query = $this->entityTypeManager
      ->getStorage('nodehive_space')
      ->getQuery()
      ->accessCheck(TRUE);
    $spaces_id = $spaces_query->execute();

    // If user has no spaces, return empty result
    if (empty($spaces_id)) {
      return [];
    }

    // Show nodes from user's spaces
    $query->condition("nodehive_space", $spaces_id, "IN");

    foreach ($conditions as $field => $value) {
      $query->condition($field, $value);
    }
    return $this->entityTypeManager->getStorage('node')->loadMultiple($query->execute());
  }

}

