<?php

namespace Drupal\nodehive_core\Helper;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Url;

class NodehiveLanguageHelper {
  /**
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  public function __construct(LanguageManagerInterface $language_manager) {
    $this->languageManager = $language_manager;
  }

  /**
   * Builds language edit/add links for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to build links for.
   *
   * @return array
   *   A render array of language links.
   */
  public function buildLanguageEditLinks(ContentEntityInterface $entity) {
    $build = [];
    $langcodes = array_keys($this->languageManager->getLanguages());

    $links = [];
    foreach ($langcodes as $langcode) {
      if ($entity->hasTranslation($langcode)) {
        $links[$langcode] = [
          '#title' => $langcode,
          '#type' => 'link',
          '#url' => $entity->getTranslation($langcode)->toUrl('edit-form'),
          '#attributes' => [
            'class' => [
              $langcode . '-has-translation',
              'language-has-translation',
            ],
            'title' => 'Edit ' . $langcode . ' translation',
          ],
        ];
      } else {
        $links[$langcode] = [
          '#title' => $langcode,
          '#type' => 'link',
          '#url' => Url::fromRoute('entity.' . $entity->getEntityTypeId() . '.content_translation_add', [
            'source' => $entity->language()->getId(),
            'target' => $langcode,
            $entity->getEntityTypeId() => $entity->id(),
          ]),
          '#attributes' => [
            'class' => [
              'language-add-translation',
            ],
            'title' => 'Add ' . $langcode . ' translation',
          ],
        ];
      }
    }

    $build['translation_link_wrapper'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['flex', 'gap-1'],
      ],
    ];
    foreach ($links as $langcode => $link) {
      $build['translation_link_wrapper'][$langcode] = $link;
    }

    $build['translation_link_wrapper']['#attached']['library'][] = 'nodehive_core/views.entity.translations.links';

    return $build;
  }

  /**
   * Builds language view/create links for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to build links for.
   *
   * @return array
   *   A render array of language links.
   */
  public function buildLanguageViewLinks(ContentEntityInterface $entity) {
    $build = [];
    $langcodes = array_keys($this->languageManager->getLanguages());

    $links = [];
    foreach ($langcodes as $langcode) {
      if ($entity->hasTranslation($langcode)) {
        $translation = $entity->getTranslation($langcode);
        $links[$langcode] = [
          '#title' => $langcode,
          '#type' => 'link',
          '#url' => $translation->toUrl(),
          '#attributes' => [
            'class' => [
              $langcode . '-has-translation',
              'language-has-translation',
            ],
            'title' => 'View ' . $langcode . ' translation',
          ],
        ];
      } else {
        $links[$langcode] = [
          '#title' => $langcode,
          '#type' => 'link',
          '#url' => Url::fromRoute('entity.' . $entity->getEntityTypeId() . '.content_translation_add', [
            'source' => $entity->language()->getId(),
            'target' => $langcode,
            $entity->getEntityTypeId() => $entity->id(),
          ]),
          '#attributes' => [
            'class' => [
              'language-add-translation',
            ],
            'title' => 'Create ' . $langcode . ' translation',
          ],
        ];
      }
    }

    $build['translation_link_wrapper'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['flex', 'gap-1'],
      ],
    ];
    foreach ($links as $langcode => $link) {
      $build['translation_link_wrapper'][$langcode] = $link;
    }

    $build['translation_link_wrapper']['#attached']['library'][] = 'nodehive_core/views.entity.translations.links';

    return $build;
  }
}
