<?php

namespace Drupal\nodehive_core;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\nodehive_core\Helper\NodehiveLanguageHelper;

/**
 * Service for preprocessing node variables with multiple space iframe support.
 */
class NodePreprocessorMultipleSpaceIframe {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new NodePreprocessorMultipleSpaceIframe object.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    AccountProxyInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    RouteMatchInterface $route_match,
    ConfigFactoryInterface $config_factory
  ) {
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->routeMatch = $route_match;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('current_route_match'),
      $container->get('config.factory')
    );
  }

  /**
   * Preprocesses node variables.
   *
   * @param array &$variables
   *   The variables array.
   */
  public function preprocess(array &$variables) {
    /** @var \Drupal\node\NodeInterface $node */
    $node = $variables['node'];
    if (!$node->hasField("nodehive_space")) {
      return;
    }

    // Check if any of the target roles exist in the user's roles
    $access_space = $this->currentUser->hasPermission('administer nodehive space');

    if ($access_space && ($variables['view_mode'] == 'default' || $variables['view_mode'] == 'full')) {
      $this->addNodehiveSidebar($variables, $node);
      $this->addPreviewTabs($variables, $node);
    }

    $variables['#attached']['library'][] = "nodehive_core/node";
    $variables['#attached']['library'][] = 'nodehive_core/nodehive-preview-tabs';
    $this->addPreviewTabsScript($variables);
  }

  /**
   * Adds the NodeHive sidebar to the variables.
   *
   * @param array &$variables
   *   The variables array.
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  protected function addNodehiveSidebar(array &$variables, NodeInterface $node) {
    $variables["content"]["nodehive_sidebar"] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ["nodehive-sidebar", "flex", "gap-4", "bg-white", "p-4", "rounded-lg", "shadow-md", "smaller"],
      ],
    ];

    $this->addNodeBasicInfo($variables, $node);
    $this->addLanguageSwitcher($variables, $node);
  }

  /**
   * Adds basic node information to the sidebar.
   *
   * @param array &$variables
   *   The variables array.
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  protected function addNodeBasicInfo(array &$variables, NodeInterface $node) {
    $author = $node->getOwner();
    $created = \Drupal::service('date.formatter')->format($node->getCreatedTime(), 'short');
    $changed = \Drupal::service('date.formatter')->format($node->getChangedTime(), 'short');
    $type_label = $node->type->entity ? $node->type->entity->label() : $node->bundle();

    $variables["content"]["nodehive_sidebar"]["node_basic_info"] = [
      '#type' => 'fieldset',
      '#title' => t('Content Information'),
      '#attributes' => [
        'class' => ['nodehive-node-basic-info'],
      ],
      '#weight' => -100,
      'content' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['flex', 'flex-col', 'gap-2', 'py-1', 'w-[400px]'],
        ],
        'title' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['truncate', 'flex', 'justify-between', 'text-xs', 'text-gray-700', 'gap-4']],
          'label' => ['#markup' => t('Title:')],
          'value' => ['#markup' => '<span class="font-medium truncate>' . Html::escape($node->label()) . '</span>'],
        ],
        'type' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['flex', 'justify-between', 'text-xs', 'text-gray-700', 'gap-4']],
          'label' => ['#markup' => t('Type:')],
          'value' => ['#markup' => '<span class="font-medium truncate">' . Html::escape($type_label) . '</span>'],
        ],
      ],
    ];
  }

  /**
   * Adds language switcher to the sidebar.
   *
   * @param array &$variables
   *   The variables array.
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  protected function addLanguageSwitcher(array &$variables, NodeInterface $node) {
    $variables["content"]["nodehive_sidebar"]["language_switcher"] = [
      '#type' => 'fieldset',
      '#title' => t('Translations'),
      '#attributes' => [
        'class' => ['nodehive-language-switcher'],
      ],
      '#weight' => -99,
      'content' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['flex', 'flex-col', 'gap-2', 'py-1', 'w-[400px]'],
        ],
      ],
    ];

    $languageHelper = \Drupal::service('nodehive_core.language_helper');
    $variables["content"]["nodehive_sidebar"]["language_switcher"]["content"] = $languageHelper->buildLanguageViewLinks($node);
    $variables["content"]["nodehive_sidebar"]["language_switcher"]['#attached']['library'][] = 'nodehive_core/views.entity.translations.links';
  }

  /**
   * Adds preview tabs to the variables.
   *
   * @param array &$variables
   *   The variables array.
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  protected function addPreviewTabs(array &$variables, NodeInterface $node) {
    $variables["content"]["nodehive_preview_tabs"] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ["nodehive-preview-tabs"],
        'style' => 'margin-top: 20px; width: 100%;',
      ],
    ];

    $this->addPreviewTabsNavigation($variables);
    $this->addPreviewTabsContent($variables, $node);
  }

  /**
   * Adds preview tabs navigation.
   *
   * @param array &$variables
   *   The variables array.
   */
  protected function addPreviewTabsNavigation(array &$variables) {
    $variables["content"]["nodehive_preview_tabs"]["tabs"] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ["nodehive-preview-tabs-nav"],
        'style' => 'display: flex; gap: 10px; margin-bottom:-2px;',
      ],
    ];

    $variables["content"]["nodehive_preview_tabs"]["tabs"]["structure"] = [
      '#type' => 'html_tag',
      '#tag' => 'button',
      '#value' => t('Structured Content View'),
      '#attributes' => [
        'class' => ['nodehive-preview-tab-button', 'rounded-t-lg'],
        'data-space-id' => 'structure',
        'style' => 'padding: 8px 16px; border: 1px solid #ccc; background: #f5f5f5; cursor: pointer;',
      ],
    ];
  }

  /**
   * Adds preview tabs content.
   *
   * @param array &$variables
   *   The variables array.
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  protected function addPreviewTabsContent(array &$variables, NodeInterface $node) {
    $variables["content"]["nodehive_preview_tabs"]["previews"] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ["nodehive-preview-tabs-content"],
        'style' => 'width: 100%;',
      ],
    ];

    $variables["content"]["nodehive_preview_tabs"]["previews"]["structure"] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ["nodehive-preview-tab-content"],
        'data-space-id' => 'structure',
        'style' => 'display: block;',
      ],
    ];

    $this->addSpacePreviews($variables, $node);
  }

  /**
   * Adds space previews to the variables.
   *
   * @param array &$variables
   *   The variables array.
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  protected function addSpacePreviews(array &$variables, NodeInterface $node) {
    $spaces = $node->get('nodehive_space')->referencedEntities();
    if (empty($spaces)) {
      return;
    }

    foreach ($spaces as $space) {
      $space_url = $space->get('space_url')->getString();
      if (empty($space_url)) {
        continue;
      }

      $preview_url = $space_url . '/node/' . $node->id();
      $internalPath = $node->toUrl()->toString();
      $open_url = $space_url . '/node/' . $node->id();

      $this->addSpacePreviewTab($variables, $space, $preview_url, $internalPath, $open_url);
    }
  }

  /**
   * Adds a space preview tab.
   *
   * @param array &$variables
   *   The variables array.
   * @param \Drupal\nodehive_core\SpaceInterface $space
   *   The space entity.
   * @param string $preview_url
   *   The preview URL.
   * @param string $internalPath
   *   The internal path.
   * @param string $open_url
   *   The open URL.
   */
  protected function addSpacePreviewTab(array &$variables, $space, $preview_url, $internalPath, $open_url) {
    $space_id = $space->id();
    $space_label = $space->label();

    $variables["content"]["nodehive_preview_tabs"]["tabs"][$space_id] = [
      '#type' => 'html_tag',
      '#tag' => 'button',
      '#value' => $space_label,
      '#attributes' => [
        'class' => ['nodehive-preview-tab-button', 'rounded-t-lg'],
        'data-space-id' => $space_id,
        'style' => 'padding: 8px 16px; border: 1px solid #ccc; background: #f5f5f5; cursor: pointer;',
      ],
    ];

    $variables["content"]["nodehive_preview_tabs"]["previews"][$space_id] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ["nodehive-preview-tab-content"],
        'data-space-id' => $space_id,
        'style' => 'display: none;',
      ],
      'iframe' => [
        '#type' => 'html_tag',
        '#tag' => 'iframe',
        '#attributes' => [
          'src' => $preview_url,
          'style' => 'width: 100%; height: 800px; border: none;',
          'data-internal-path' => $internalPath,
          'data-open-url' => $open_url,
        ],
      ],
    ];
  }

  /**
   * Adds preview tabs script to the variables.
   *
   * @param array &$variables
   *   The variables array.
   */
  protected function addPreviewTabsScript(array &$variables) {
    $variables['#attached']['library'][] = 'nodehive_core/nodehive-preview-tabs';
  }
}
