<?php

namespace Drupal\nodehive_core;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Drupal\nodehive_core\Helper\NodehiveLanguageHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Html;

class StructuredContentNodePreprocessor
{
  protected $currentUser;
  protected $entityTypeManager;
  protected $languageManager;
  protected $languageHelper;

  public function __construct(
    AccountProxyInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    NodehiveLanguageHelper $language_helper
  ) {
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->languageHelper = $language_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('nodehive_core.language_helper')
    );
  }

  public function preprocess(array &$variables)
  {
    /** @var \Drupal\node\NodeInterface $node */
    $node = $variables['node'];
    if (!$node->hasField("nodehive_space")) {
      return;
    }

    // Check if any of the target roles exist in the user's roles
    $access_space = $this->currentUser->hasPermission('administer nodehive space');

    if ($access_space && ($variables['view_mode'] == 'default' || $variables['view_mode'] == 'full')) {
      // Add NodeHive Context Links.
      $variables["content"]["nodehive_sidebar"] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ["nodehive-sidebar", "flex", "gap-4", "smaller", 'mb-4'],
        ],
      ];


      $author = $node->getOwner();
      $created = \Drupal::service('date.formatter')->format($node->getCreatedTime(), 'short');
      $changed = \Drupal::service('date.formatter')->format($node->getChangedTime(), 'short');
      $type_label = $node->type->entity ? $node->type->entity->label() : $node->bundle();

      $variables["content"]["nodehive_sidebar"]["node_basic_info"] = [
        '#type' => 'fieldset',
        '#title' => t('Content Information'),
        '#attributes' => [
          'class' => ['nodehive-node-basic-info'],
          'style' => 'background: white'
        ],
        '#weight' => -100,
        'content' => [
          '#type' => 'container',
          '#attributes' => [
            'class' => ['flex', 'flex-col', 'gap-2', 'py-1', 'w-[400px]'],
          ],
          'title' => [
            '#type' => 'container',
            '#attributes' => ['class' => ['truncate', 'flex', 'justify-between', 'text-xs', 'text-gray-700', 'gap-4']],
            'label' => ['#markup' => t('Title:')],
            'value' => ['#markup' => '<span class="font-medium truncate>' . Html::escape($node->label()) . '</span>'],
          ],
          'type' => [
            '#type' => 'container',
            '#attributes' => ['class' => ['flex', 'justify-between', 'text-xs', 'text-gray-700', 'gap-4']],
            'label' => ['#markup' => t('Type:')],
            'value' => ['#markup' => '<span class="font-medium truncate">' . Html::escape($type_label) . '</span>'],
          ],
          'translations' => [
            '#type' => 'container',
            '#attributes' => ['class' => ['flex', 'justify-between', 'text-xs', 'text-gray-700', 'gap-4']],
            'label' => ['#markup' => t('Translations:')],
            'value' => $this->languageHelper->buildLanguageViewLinks($node),
          ],
        ],
      ];

      // Nodehive language switcher.
      /*
      $variables["content"]["nodehive_sidebar"]["nodehive_lang_switcher"] = [
        '#type' => 'fieldset',
        '#title' => t('Translations'),
        '#attributes' => [
          'class' => ["nodehive-language-switcher"],
          'style' => 'background: white'
        ],
      ];

      $variables["content"]["nodehive_sidebar"]["nodehive_lang_switcher"]["content"] = $this->languageHelper->buildLanguageViewLinks($node);
      $variables["content"]["nodehive_sidebar"]["nodehive_lang_switcher"]['#attached']['library'][] = 'nodehive_core/views.entity.translations.links';
      */

      // Nodehive Spaces List.
      $variables["content"]["nodehive_sidebar"]["spaces"] = [
        '#type' => 'div',
        '#attributes' => [
          'class' => ["nodehive-spaces"],
          'style' => 'display: flex; justify-content: space-between; gap:20px;'
        ],
      ];
      $spaces = $node->get("nodehive_space")->getValue();

      foreach ($spaces as $space_ref) {
        /** @var \Drupal\nodehive_core\SpaceInterface $space */
        $space = $this->entityTypeManager->getStorage("nodehive_space")->load($space_ref['target_id']);
        $space_urls = [];
        foreach ($space->space_url->getValue() as $value) {
          $space_urls[] = $value['uri'];
        }

        $preview_url = $space_urls[0] ?? FALSE;
        $urlObject = $node->toUrl();
        $internalPath = $urlObject->toString();
        $query_parameters = ['url' => $preview_url . $internalPath];

        // Prepare Nodehive Visual Editor link.
        $visual_editor_url = Url::fromRoute("nodehive_visualeditor.visual_editor_single", ["nodehive_space" => $space->id()])->setOption('query', $query_parameters);
        $visual_editor_url_string = $visual_editor_url->toString();
        // Prepare Open Website link.
        $open_url = $preview_url . $internalPath;

        $space_overview_url = Url::fromRoute("entity.nodehive_space.canonical", ["nodehive_space" => $space->id()])->toString();

        // Fieldset for each space
        $variables["content"]["nodehive_sidebar"]["spaces"][$space->id()] = [
          '#type' => 'fieldset',
          '#title' => '<div class="flex gap-4">Space: ' . $space->label() . ' <a href="' . $space_overview_url . '" class="text-xs">Overview</a></div>',
          '#attributes' => [
            'style' => 'flex: 1;border-left: 2px solid var(--gin-color-primary); padding-left: 10px; background: white',
          ],
          'content' => [
            /*
            'view_space' => [
              '#type' => 'html_tag',
              '#tag' => 'a',
              '#attributes' => [
                'href' => Url::fromRoute("entity.nodehive_space.canonical", ["nodehive_space" => $space->id()])->toString(),
                'class' => ['button', 'button--small'],
                'style' => 'flex: 1; text-align: center;',
              ],
              '#value' => '<- Overview',
            ],
            */
            'visual_editor_link' => [
              '#type' => 'html_tag',
              '#tag' => 'a',
              '#attributes' => [
                'href' => $visual_editor_url->toString(),
                'class' => ['m-0', 'button', 'button--extrasmall', 'button-'],
              ],
              '#value' => 'Open Editor',
            ],

            'external_link' => [
              '#type' => 'html_tag',
              '#tag' => 'a',
              '#attributes' => [
                'href' => $open_url,
                'class' => ['button', 'button--extrasmall'],
                'target' => '_blank'
              ],
              '#value' => 'Visit ↗',
            ],

          ],
        ];
      }


      // Show Content Type Configuration
      $current_url = Url::fromRoute('<current>');
      $destination = $current_url->toString();
      $node_type = $node->bundle();
      $node_type_entity = $this->entityTypeManager->getStorage('node_type')->load($node_type);

      if ($node_type_entity) {
        $user = $this->currentUser;
        if (
          $user->hasPermission('administer content types')
          || $user->hasPermission('administer node display')
          || $user->hasPermission('administer node fields')
          || $user->hasPermission('administer node form display')
        ) {
          $operations = [];

          // Add 'Manage Fields'
          if ($user->hasPermission('administer node fields')) {
            $operations['manage_fields'] = [
              'title' => t('Manage fields'),
              'url' => Url::fromRoute('entity.node.field_ui_fields', [
                'node_type' => $node_type_entity->id()
              ], ['query' => ['destination' => $destination]])
            ];
          }

          if ($user->hasPermission('administer node display')) {
            // Add 'Manage Display'
            $operations['manage_display'] = [
              'title' => t('Manage display'),
              'url' => Url::fromRoute('entity.entity_view_display.node.default', [
                'node_type' => $node_type_entity->id(),
                'view_mode_name' => 'default',
              ], ['query' => ['destination' => $destination]])
            ];
          }

          // Add 'Manage Form Display'
          if ($user->hasPermission('administer node form display')) {
            $operations['manage_form'] = [
              'title' => t('Manage form display'),
              'url' => Url::fromRoute('entity.entity_form_display.node.default', [
                'node_type' => $node_type_entity->id(),
                'form_mode_name' => 'default',
              ], ['query' => ['destination' => $destination]])
            ];
          }

          // Add 'Edit Content Type'
          if ($user->hasPermission('administer content types')) {
            $operations['edit'] = [
              'title' => t('Manage settings'),
              'url' => Url::fromRoute('entity.node_type.edit_form', [
                'node_type' => $node_type_entity->id()
              ], ['query' => ['destination' => $destination]])
            ];
          }

          $links = [];
          foreach ($operations as $operation_id => $operation) {
            if (!in_array($operation_id, ['delete'])) {  // Excluding 'delete'
              // Append the destination to the URL
              $operation['url']->setOption('query', ['destination' => $destination]);
              $links[$operation_id] = [
                'title' => $operation['title'],
                'url' => $operation['url'],
              ];
            }
          }

          if (!empty($links)) {
            $variables['content']['nodehive_sidebar']['content_type_operations_container'] = [
              '#type' => 'fieldset',
              '#title' => t('Configure ') . $node_type_entity->label(),
              '#attributes' => [
                'class' => ['nodehive-content-type-operations-container'],
                'style' => 'background: white'
              ],
              '#weight' => 55,
            ];

            $variables['content']['nodehive_sidebar']['content_type_operations_container']['div'] = [
              '#type' => 'html_tag',
              '#tag' => 'div',
              '#attributes' => ['class' => ['gin-utility-margin-small']],
            ];
            $variables['content']['nodehive_sidebar']['content_type_operations_container']['div']['dropbutton'] = [
              '#type' => 'dropbutton',
              '#links' => $links,
              '#attributes' => ['class' => ['dropbutton', 'dropbutton--extrasmall', 'dropbutton--multiple', 'dropbutton--gin']],
            ];
          }
        }
      }
    }

    $variables['#attached']['library'][] = "nodehive_core/node";
  }
}
