<?php

namespace Drupal\nonce_generator\Element;

use Drupal\Core\Render\Element\RenderElementBase;

/**
 * Provides a nonce script render element.
 *
 * This render element uses lazy builders to ensure that nonce-enabled
 * scripts are generated with fresh nonces on each request, even when
 * the page is cached.
 *
 * Usage:
 * @code
 * $build['script'] = [
 *   '#type' => 'nonce_script',
 *   '#plugin_id' => 'piwik',
 * ];
 * @endcode
 *
 * @RenderElement("nonce_script")
 */
class NonceScript extends RenderElementBase {

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    $class = static::class;
    return [
      '#pre_render' => [[$class, 'preRenderNonceScript']],
      '#plugin_id' => NULL,
      '#all_plugins' => FALSE,
    ];
  }

  /**
   * Pre-render callback for nonce script elements.
   *
   * @param array $element
   *   The render element.
   *
   * @return array
   *   The modified render element.
   */
  public static function preRenderNonceScript(array $element): array {
    if (!empty($element['#all_plugins'])) {
      $child = [
        '#lazy_builder' => ['nonce_generator.lazy_builder:buildAllScripts', []],
        '#create_placeholder' => TRUE,
        '#cache' => ['max-age' => 0],
      ];
    }
    elseif (!empty($element['#plugin_id'])) {
      $child = [
        '#lazy_builder' => [
          'nonce_generator.lazy_builder:buildScript',
          [$element['#plugin_id']],
        ],
        '#create_placeholder' => TRUE,
        '#cache' => ['max-age' => 0],
      ];
    }
    else {
      return [];
    }

    $element['content'] = $child;

    return $element;
  }

}
